// Background Service Worker for Traffic Management Assistant

// Open side panel when extension icon is clicked
chrome.sidePanel.setPanelBehavior({ openPanelOnActionClick: true });

// Track the current tab URL and notify side panel of changes
let currentTabUrl = '';

// Listen for tab URL updates
chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
  // Only process if URL changed and it's a Google Maps URL
  if (changeInfo.url && isGoogleMapsUrl(changeInfo.url)) {
    notifySidePanel(changeInfo.url);
  }
});

// Listen for tab activation (user switches tabs)
chrome.tabs.onActivated.addListener(async (activeInfo) => {
  try {
    const tab = await chrome.tabs.get(activeInfo.tabId);
    if (tab.url && isGoogleMapsUrl(tab.url)) {
      notifySidePanel(tab.url);
    } else {
      notifySidePanel(null);
    }
  } catch (error) {
    console.error('Error getting tab info:', error);
  }
});

// Check if URL is a Google Maps URL
function isGoogleMapsUrl(url) {
  return url && (
    url.includes('google.com/maps') ||
    url.includes('maps.google.com')
  );
}

// Send URL update to side panel
function notifySidePanel(url) {
  chrome.runtime.sendMessage({
    type: 'URL_UPDATE',
    url: url
  }).catch(() => {
    // Side panel might not be open, ignore error
  });
}

// Handle messages from side panel
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.type === 'GET_CURRENT_TAB_URL') {
    getCurrentTabUrl().then(url => {
      sendResponse({ url: url });
    });
    return true; // Keep channel open for async response
  }
});

// Get the current active tab's URL
async function getCurrentTabUrl() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (tab && tab.url && isGoogleMapsUrl(tab.url)) {
      return tab.url;
    }
    return null;
  } catch (error) {
    console.error('Error getting current tab URL:', error);
    return null;
  }
}
