// Side Panel JavaScript for Traffic Management Assistant

// Worker endpoint for Street View image proxy
const WORKER_URL = 'https://sv-proxy-8f3k.agenticintelligence.workers.dev';

// ==================== ZERO TRUST AUTH ====================

// Wrapper for fetch that handles Cloudflare Zero Trust auth
async function authFetch(url, options = {}) {
  // Include credentials to send CF_Authorization cookie
  options.credentials = 'include';
  
  try {
    const response = await fetch(url, options);
    
    // Check for opaque redirect (CORS blocked redirect to Zero Trust)
    if (response.type === 'opaqueredirect') {
      showAuthRequired();
      throw new Error('AUTH_REQUIRED');
    }
    
    // Check response content for Zero Trust login page
    if (response.status === 200) {
      const contentType = response.headers.get('content-type') || '';
      // If we expected JSON/image but got HTML, it's probably the login page
      if (contentType.includes('text/html') && !url.includes('.html')) {
        const text = await response.text();
        if (text.includes('cloudflareaccess') || text.includes('Access denied') || text.includes('cf-access')) {
          showAuthRequired();
          throw new Error('AUTH_REQUIRED');
        }
        // Return a new response with the text we already read
        return new Response(text, { status: response.status, headers: response.headers });
      }
    }
    
    // Check for 401/403 responses  
    if (response.status === 401 || response.status === 403) {
      showAuthRequired();
      throw new Error('AUTH_REQUIRED');
    }
    
    return response;
  } catch (error) {
    // Network errors or CORS issues when Zero Trust redirects
    if (error.message === 'AUTH_REQUIRED') throw error;
    if (error.message === 'Failed to fetch' || error.name === 'TypeError') {
      // Likely a CORS issue from Zero Trust redirect - show auth
      showAuthRequired();
      throw new Error('AUTH_REQUIRED');
    }
    throw error;
  }
}

// Show auth required UI
function showAuthRequired() {
  // Check if auth modal already exists
  if (document.getElementById('auth-modal')) return;
  
  const modal = document.createElement('div');
  modal.id = 'auth-modal';
  modal.innerHTML = `
    <div style="position: fixed; inset: 0; background: rgba(0,0,0,0.7); display: flex; align-items: center; justify-content: center; z-index: 10000;">
      <div style="background: var(--bg-primary, #1a1a2e); border: 1px solid var(--border-color, #333); border-radius: 12px; padding: 24px; max-width: 400px; text-align: center;">
        <h3 style="margin: 0 0 12px; color: var(--text-primary, #fff);">Authentication Required</h3>
        <p style="margin: 0 0 20px; color: var(--text-secondary, #aaa); font-size: 14px;">
          Sign in with your work email to access the Street View proxy.
        </p>
        <button id="auth-login-btn" style="background: var(--accent-color, #4a90d9); color: white; border: none; padding: 12px 24px; border-radius: 8px; cursor: pointer; font-size: 14px; margin-right: 8px;">
          Sign In
        </button>
        <button id="auth-dismiss-btn" style="background: transparent; color: var(--text-secondary, #aaa); border: 1px solid var(--border-color, #333); padding: 12px 24px; border-radius: 8px; cursor: pointer; font-size: 14px;">
          Dismiss
        </button>
      </div>
    </div>
  `;
  document.body.appendChild(modal);
  
  document.getElementById('auth-login-btn').addEventListener('click', () => {
    // Open the worker URL directly - Zero Trust will redirect to login
    window.open(WORKER_URL, '_blank');
    modal.remove();
  });
  
  document.getElementById('auth-dismiss-btn').addEventListener('click', () => {
    modal.remove();
  });
}
const DB_NAME = 'TrafficManagementDB';
const DB_VERSION = 1;
const STORE_NAME = 'images';

// Global state
let db = null;
let currentParams = null;
let currentImageBlob = null;
let currentBatch = null;
let cropSelection = null;
let isCropMode = false;

// DOM Elements - Capture Tab
const statusIndicator = document.getElementById('status-indicator');
const statusText = document.getElementById('status-text');
const paramLat = document.getElementById('param-lat');
const paramLng = document.getElementById('param-lng');
const paramHeading = document.getElementById('param-heading');
const paramPitch = document.getElementById('param-pitch');
const paramZoom = document.getElementById('param-zoom');
const paramFov = document.getElementById('param-fov');
const paramImageryDate = document.getElementById('param-imagery-date');
const imageSection = document.getElementById('image-section');
const imageContainer = document.getElementById('image-container');
const previewImage = document.getElementById('preview-image');
const imageLoading = document.getElementById('image-loading');
const selectionOverlay = document.getElementById('selection-overlay');
const cropControls = document.getElementById('crop-controls');
const captureImageBtn = document.getElementById('capture-image-btn');
const saveToBatchBtn = document.getElementById('save-to-batch-btn');
const enableCropBtn = document.getElementById('enable-crop-btn');
const batchSelect = document.getElementById('batch-select');
const batchInfo = document.getElementById('batch-info');
const batchImageCount = document.getElementById('batch-image-count');

// DOM Elements - Gallery Tab
const galleryBatchFilter = document.getElementById('gallery-batch-filter');
const uploadBatchBtn = document.getElementById('upload-batch-btn');
const deleteBatchBtn = document.getElementById('delete-batch-btn');
const uploadProgress = document.getElementById('upload-progress');
const progressFill = document.getElementById('progress-fill');
const progressText = document.getElementById('progress-text');
const galleryGrid = document.getElementById('gallery-grid');
const galleryCount = document.getElementById('gallery-count');
const gallerySize = document.getElementById('gallery-size');

// DOM Elements - Settings Tab
const roboflowApiKey = document.getElementById('roboflow-api-key');
const roboflowProject = document.getElementById('roboflow-project');
const roboflowWorkspace = document.getElementById('roboflow-workspace');
const roboflowUrlPaste = document.getElementById('roboflow-url-paste');
const toggleKeyVisibility = document.getElementById('toggle-key-visibility');
const saveSettingsBtn = document.getElementById('save-settings-btn');
const settingsStatus = document.getElementById('settings-status');
const optimizeImages = document.getElementById('optimize-images');
const imageQuality = document.getElementById('image-quality');
const qualityValue = document.getElementById('quality-value');
const maxDimension = document.getElementById('max-dimension');
const exportDataBtn = document.getElementById('export-data-btn');
const clearDataBtn = document.getElementById('clear-data-btn');

// DOM Elements - Modals
const newBatchModal = document.getElementById('new-batch-modal');
const newBatchName = document.getElementById('new-batch-name');
const createBatchBtn = document.getElementById('create-batch-btn');
const cancelBatchBtn = document.getElementById('cancel-batch-btn');
const imageModal = document.getElementById('image-modal');
const modalImage = document.getElementById('modal-image');
const modalDeleteBtn = document.getElementById('modal-delete-btn');
const modalCloseBtn = document.getElementById('modal-close-btn');

// ==================== INITIALIZATION ====================

document.addEventListener('DOMContentLoaded', async () => {
  await initDB();
  await loadSettings();
  await loadBatches();
  setupEventListeners();
  requestCurrentUrl();
});

// ==================== INDEXEDDB ====================

async function initDB() {
  return new Promise((resolve, reject) => {
    const request = indexedDB.open(DB_NAME, DB_VERSION);

    request.onerror = () => reject(request.error);
    request.onsuccess = () => {
      db = request.result;
      resolve(db);
    };

    request.onupgradeneeded = (event) => {
      const database = event.target.result;
      if (!database.objectStoreNames.contains(STORE_NAME)) {
        const store = database.createObjectStore(STORE_NAME, { keyPath: 'id', autoIncrement: true });
        store.createIndex('batch', 'batch', { unique: false });
        store.createIndex('timestamp', 'timestamp', { unique: false });
      }
    };
  });
}

async function saveImageToDB(imageData) {
  return new Promise((resolve, reject) => {
    const transaction = db.transaction([STORE_NAME], 'readwrite');
    const store = transaction.objectStore(STORE_NAME);
    const request = store.add(imageData);
    request.onsuccess = () => resolve(request.result);
    request.onerror = () => reject(request.error);
  });
}

async function getImagesByBatch(batchName) {
  return new Promise((resolve, reject) => {
    const transaction = db.transaction([STORE_NAME], 'readonly');
    const store = transaction.objectStore(STORE_NAME);
    const index = store.index('batch');
    const request = batchName === 'all' ? store.getAll() : index.getAll(batchName);
    request.onsuccess = () => resolve(request.result);
    request.onerror = () => reject(request.error);
  });
}

async function getAllImages() {
  return new Promise((resolve, reject) => {
    const transaction = db.transaction([STORE_NAME], 'readonly');
    const store = transaction.objectStore(STORE_NAME);
    const request = store.getAll();
    request.onsuccess = () => resolve(request.result);
    request.onerror = () => reject(request.error);
  });
}

async function deleteImageFromDB(id) {
  return new Promise((resolve, reject) => {
    const transaction = db.transaction([STORE_NAME], 'readwrite');
    const store = transaction.objectStore(STORE_NAME);
    const request = store.delete(id);
    request.onsuccess = () => resolve();
    request.onerror = () => reject(request.error);
  });
}

async function deleteImagesByBatch(batchName) {
  const images = await getImagesByBatch(batchName);
  const transaction = db.transaction([STORE_NAME], 'readwrite');
  const store = transaction.objectStore(STORE_NAME);
  for (const img of images) {
    store.delete(img.id);
  }
  return new Promise((resolve, reject) => {
    transaction.oncomplete = () => resolve();
    transaction.onerror = () => reject(transaction.error);
  });
}

async function clearAllImages() {
  return new Promise((resolve, reject) => {
    const transaction = db.transaction([STORE_NAME], 'readwrite');
    const store = transaction.objectStore(STORE_NAME);
    const request = store.clear();
    request.onsuccess = () => resolve();
    request.onerror = () => reject(request.error);
  });
}

// ==================== SETTINGS ====================

async function loadSettings() {
  const result = await chrome.storage.local.get([
    'roboflow_api_key',
    'roboflow_project',
    'roboflow_workspace',
    'optimize_images',
    'image_quality',
    'max_dimension',
    'batches'
  ]);

  if (result.roboflow_api_key) roboflowApiKey.value = result.roboflow_api_key;
  if (result.roboflow_project) roboflowProject.value = result.roboflow_project;
  if (result.roboflow_workspace) roboflowWorkspace.value = result.roboflow_workspace;
  optimizeImages.checked = result.optimize_images !== false;
  if (result.image_quality) {
    imageQuality.value = result.image_quality;
    qualityValue.textContent = Math.round(result.image_quality * 100) + '%';
  }
  if (result.max_dimension) maxDimension.value = result.max_dimension;
}

async function saveSettings() {
  await chrome.storage.local.set({
    roboflow_api_key: roboflowApiKey.value,
    roboflow_project: roboflowProject.value,
    roboflow_workspace: roboflowWorkspace.value,
    optimize_images: optimizeImages.checked,
    image_quality: parseFloat(imageQuality.value),
    max_dimension: parseInt(maxDimension.value)
  });

  showSettingsStatus('Settings saved successfully', 'success');
}

function showSettingsStatus(message, type) {
  settingsStatus.textContent = message;
  settingsStatus.className = `settings-status ${type}`;
  settingsStatus.style.display = 'block';
  setTimeout(() => {
    settingsStatus.style.display = 'none';
  }, 3000);
}

// ==================== BATCH MANAGEMENT ====================

async function loadBatches() {
  const result = await chrome.storage.local.get(['batches', 'currentBatch']);
  const batches = result.batches || [];

  // Populate batch selects
  populateBatchSelect(batchSelect, batches, true);
  populateBatchSelect(galleryBatchFilter, batches, false);

  // Restore current batch
  if (result.currentBatch && batches.includes(result.currentBatch)) {
    batchSelect.value = result.currentBatch;
    currentBatch = result.currentBatch;
    updateBatchInfo();
  }
}

function populateBatchSelect(selectEl, batches, includeEmpty) {
  const currentValue = selectEl.value;
  selectEl.innerHTML = '';

  if (includeEmpty) {
    const emptyOption = document.createElement('option');
    emptyOption.value = '';
    emptyOption.textContent = '-- Select or Create Batch --';
    selectEl.appendChild(emptyOption);
  } else {
    const allOption = document.createElement('option');
    allOption.value = 'all';
    allOption.textContent = 'All Batches';
    selectEl.appendChild(allOption);
  }

  batches.forEach(batch => {
    const option = document.createElement('option');
    option.value = batch;
    option.textContent = batch;
    selectEl.appendChild(option);
  });

  if (currentValue) selectEl.value = currentValue;
}

async function createBatch(name) {
  const sanitized = sanitizeBatchName(name);
  if (!sanitized) {
    alert('Invalid batch name. Use only letters, numbers, dashes, and underscores.');
    return false;
  }

  const result = await chrome.storage.local.get(['batches']);
  const batches = result.batches || [];

  if (batches.includes(sanitized)) {
    alert('A batch with this name already exists.');
    return false;
  }

  batches.push(sanitized);
  await chrome.storage.local.set({ batches });
  await loadBatches();

  batchSelect.value = sanitized;
  currentBatch = sanitized;
  await chrome.storage.local.set({ currentBatch: sanitized });
  updateBatchInfo();

  return true;
}

function sanitizeBatchName(name) {
  if (!name) return null;
  const sanitized = name.trim().replace(/[^\w\-_.]/g, '').substring(0, 50);
  return sanitized.length > 0 ? sanitized : null;
}

async function updateBatchInfo() {
  if (!currentBatch) {
    batchInfo.style.display = 'none';
    return;
  }

  const images = await getImagesByBatch(currentBatch);
  batchImageCount.textContent = `${images.length} image${images.length !== 1 ? 's' : ''}`;
  batchInfo.style.display = 'block';
}

async function deleteBatch(batchName) {
  if (!confirm(`Delete batch "${batchName}" and all its images?`)) return;

  await deleteImagesByBatch(batchName);

  const result = await chrome.storage.local.get(['batches']);
  const batches = (result.batches || []).filter(b => b !== batchName);
  await chrome.storage.local.set({ batches });

  if (currentBatch === batchName) {
    currentBatch = null;
    await chrome.storage.local.set({ currentBatch: null });
  }

  await loadBatches();
  await renderGallery();
}

// ==================== TAB NAVIGATION ====================

function setupTabNavigation() {
  document.querySelectorAll('.tab').forEach(tab => {
    tab.addEventListener('click', () => {
      const targetTab = tab.dataset.tab;

      // Update tab buttons
      document.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
      tab.classList.add('active');

      // Update tab content
      document.querySelectorAll('.tab-content').forEach(content => content.classList.remove('active'));
      document.getElementById(`tab-${targetTab}`).classList.add('active');

      // Refresh gallery when switching to gallery tab
      if (targetTab === 'gallery') {
        renderGallery();
      }
    });
  });
}

// ==================== EVENT LISTENERS ====================

function setupEventListeners() {
  setupTabNavigation();

  // URL updates from background
  chrome.runtime.onMessage.addListener((message) => {
    if (message.type === 'URL_UPDATE') {
      handleUrlUpdate(message.url);
    }
  });

  // Capture tab
  captureImageBtn.addEventListener('click', captureImage);
  saveToBatchBtn.addEventListener('click', saveCurrentImageToBatch);
  enableCropBtn.addEventListener('click', toggleCropMode);
  document.getElementById('cancel-crop-btn').addEventListener('click', cancelCrop);
  document.getElementById('confirm-crop-btn').addEventListener('click', confirmCrop);
  batchSelect.addEventListener('change', handleBatchChange);
  document.getElementById('new-batch-btn').addEventListener('click', () => showModal(newBatchModal));

  // Crop mouse events
  imageContainer.addEventListener('mousedown', startCrop);
  imageContainer.addEventListener('mousemove', updateCrop);
  imageContainer.addEventListener('mouseup', endCrop);
  imageContainer.addEventListener('mouseleave', endCrop);

  // Gallery tab
  galleryBatchFilter.addEventListener('change', renderGallery);
  uploadBatchBtn.addEventListener('click', uploadToRoboflow);
  deleteBatchBtn.addEventListener('click', () => {
    const batch = galleryBatchFilter.value;
    if (batch && batch !== 'all') deleteBatch(batch);
  });

  // Settings tab
  toggleKeyVisibility.addEventListener('click', () => {
    const isPassword = roboflowApiKey.type === 'password';
    roboflowApiKey.type = isPassword ? 'text' : 'password';
    toggleKeyVisibility.textContent = isPassword ? 'Hide' : 'Show';
  });
  saveSettingsBtn.addEventListener('click', saveSettings);
  imageQuality.addEventListener('input', () => {
    qualityValue.textContent = Math.round(imageQuality.value * 100) + '%';
  });
  roboflowUrlPaste.addEventListener('paste', handleRoboflowUrlPaste);
  roboflowUrlPaste.addEventListener('input', handleRoboflowUrlPaste);
  exportDataBtn.addEventListener('click', exportAllData);
  clearDataBtn.addEventListener('click', clearAllData);

  // Modals
  createBatchBtn.addEventListener('click', async () => {
    if (await createBatch(newBatchName.value)) {
      hideModal(newBatchModal);
      newBatchName.value = '';
    }
  });
  cancelBatchBtn.addEventListener('click', () => hideModal(newBatchModal));
  modalCloseBtn.addEventListener('click', () => hideModal(imageModal));
  modalDeleteBtn.addEventListener('click', deleteModalImage);

  // Close modals on backdrop click
  [newBatchModal, imageModal].forEach(modal => {
    modal.addEventListener('click', (e) => {
      if (e.target === modal) hideModal(modal);
    });
  });
}

function showModal(modal) {
  modal.style.display = 'flex';
}

function hideModal(modal) {
  modal.style.display = 'none';
}

// ==================== URL PARSING ====================

function requestCurrentUrl() {
  chrome.runtime.sendMessage({ type: 'GET_CURRENT_TAB_URL' }, (response) => {
    if (response && response.url) {
      handleUrlUpdate(response.url);
    }
  });
}

function handleUrlUpdate(url) {
  if (!url) {
    setDisconnectedState();
    return;
  }

  const params = parseGoogleMapsUrl(url);

  if (params) {
    currentParams = params;
    updateDisplay(params);

    if (params.isStreetView) {
      setStreetViewState();
      captureImageBtn.disabled = false;
    } else {
      setConnectedState();
      captureImageBtn.disabled = true;
    }
  } else {
    setDisconnectedState();
  }
}

function parseGoogleMapsUrl(url) {
  if (!url) return null;

  const params = {
    lat: null, lng: null, heading: null, pitch: null, zoom: null, fov: null, isStreetView: false, imageryDate: null, panoid: null
  };

  const coordsMatch = url.match(/@(-?\d+\.?\d*),(-?\d+\.?\d*)/);
  if (!coordsMatch) return null;

  params.lat = parseFloat(coordsMatch[1]);
  params.lng = parseFloat(coordsMatch[2]);

  const afterCoordsMatch = url.match(/@-?\d+\.?\d*,-?\d+\.?\d*,([^/]+)/);
  if (afterCoordsMatch) {
    const paramString = afterCoordsMatch[1];

    const streetViewMatch = paramString.match(/(\d+\.?\d*)a/);
    if (streetViewMatch) {
      params.isStreetView = true;

      const fovMatch = paramString.match(/(\d+\.?\d*)y/);
      if (fovMatch) params.fov = parseFloat(fovMatch[1]);

      const headingMatch = paramString.match(/(\d+\.?\d*)h/);
      if (headingMatch) params.heading = parseFloat(headingMatch[1]);

      const pitchMatch = paramString.match(/(\d+\.?\d*)t/);
      if (pitchMatch) params.pitch = 90 - parseFloat(pitchMatch[1]);
    }

    const zoomMatch = paramString.match(/(\d+\.?\d*)z/);
    if (zoomMatch) params.zoom = parseFloat(zoomMatch[1]);
  }
  
  // Extract panoid from URL (format: !1s followed by the panorama ID)
  const panoidMatch = url.match(/!1s([A-Za-z0-9_-]+)/);
  if (panoidMatch) {
    params.panoid = panoidMatch[1];
  }
  
  // Note: Imagery date is no longer reliably in the URL
  // We fetch it from the metadata API instead (see fetchImageryDate)

  return params;
}

function updateDisplay(params) {
  paramLat.textContent = params.lat !== null ? params.lat.toFixed(6) : '--';
  paramLng.textContent = params.lng !== null ? params.lng.toFixed(6) : '--';
  paramHeading.textContent = params.heading !== null ? `${params.heading.toFixed(1)}°` : '--';
  paramPitch.textContent = params.pitch !== null ? `${params.pitch.toFixed(1)}°` : '--';
  paramZoom.textContent = params.zoom !== null ? params.zoom.toFixed(1) : '--';
  paramFov.textContent = params.fov !== null ? `${params.fov.toFixed(1)}°` : '--';
  paramImageryDate.textContent = params.imageryDate || '--';
  
  // If no imagery date from URL and we have coords, fetch from metadata API
  if (!params.imageryDate && params.lat !== null && params.lng !== null && params.isStreetView) {
    fetchImageryDate(params.lat, params.lng, params.panoid);
  }
}

// Fetch imagery date from Street View metadata API via worker
async function fetchImageryDate(lat, lng, panoid) {
  try {
    paramImageryDate.textContent = '...';
    const body = panoid ? { panoid } : { lat, lng };
    const response = await authFetch(`${WORKER_URL}/metadata`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(body),
    });
    
    if (response.ok) {
      const metadata = await response.json();
      if (metadata.dateFormatted) {
        paramImageryDate.textContent = metadata.dateFormatted;
        // Update currentParams so it's included in captures
        if (currentParams) {
          currentParams.imageryDate = metadata.dateFormatted;
        }
      } else {
        paramImageryDate.textContent = '--';
      }
    }
  } catch (e) {
    if (e.message === 'AUTH_REQUIRED') return; // Modal already shown
    console.error('Failed to fetch imagery date:', e);
    paramImageryDate.textContent = '--';
  }
}

function setConnectedState() {
  statusIndicator.className = 'status-indicator connected';
  statusText.textContent = 'Connected to Google Maps';
}

function setStreetViewState() {
  statusIndicator.className = 'status-indicator streetview';
  statusText.textContent = 'Street View Active';
}

function setDisconnectedState() {
  statusIndicator.className = 'status-indicator disconnected';
  statusText.textContent = 'Not on Google Maps';
  currentParams = null;
  captureImageBtn.disabled = true;
  saveToBatchBtn.disabled = true;
  enableCropBtn.disabled = true;
}

// ==================== IMAGE CAPTURE ====================

async function captureImage() {
  if (!currentParams || !currentParams.isStreetView) return;

  imageSection.style.display = 'block';
  imageLoading.classList.add('active');
  previewImage.classList.remove('loaded');
  captureImageBtn.disabled = true;
  captureImageBtn.textContent = 'Loading...';

  try {
    const response = await authFetch(WORKER_URL, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        lat: currentParams.lat,
        lng: currentParams.lng,
        heading: currentParams.heading || 0,
        pitch: currentParams.pitch || 0,
        fov: currentParams.fov || 90,
      }),
    });

    if (!response.ok) {
      const errorData = await response.json().catch(() => ({}));
      throw new Error(errorData.error || `HTTP ${response.status}`);
    }

    currentImageBlob = await response.blob();
    const imageUrl = URL.createObjectURL(currentImageBlob);

    previewImage.onload = () => {
      imageLoading.classList.remove('active');
      previewImage.classList.add('loaded');
      saveToBatchBtn.disabled = !currentBatch;
      enableCropBtn.disabled = false;
    };

    previewImage.src = imageUrl;

    captureImageBtn.textContent = 'Capture Image';
    captureImageBtn.disabled = false;

  } catch (error) {
    if (error.message === 'AUTH_REQUIRED') {
      // Modal already shown, reset UI
      imageLoading.classList.remove('active');
      captureImageBtn.textContent = 'Capture Image';
      captureImageBtn.disabled = false;
      return;
    }
    console.error('Error capturing image:', error);
    imageLoading.classList.remove('active');
    showImageError(error.message || 'Failed to capture image');
    captureImageBtn.textContent = 'Capture Image';
    captureImageBtn.disabled = false;
  }
}

function showImageError(message) {
  const existingError = imageContainer.querySelector('.image-error');
  if (existingError) existingError.remove();

  const errorDiv = document.createElement('div');
  errorDiv.className = 'image-error';
  errorDiv.textContent = message;
  imageContainer.appendChild(errorDiv);
  setTimeout(() => errorDiv.remove(), 5000);
}

// ==================== CROP FUNCTIONALITY ====================

let cropStartX, cropStartY, isCropping = false;

function toggleCropMode() {
  isCropMode = !isCropMode;
  imageContainer.classList.toggle('crop-mode', isCropMode);
  enableCropBtn.textContent = isCropMode ? 'Cancel Select' : 'Select Region';

  if (!isCropMode) {
    selectionOverlay.style.display = 'none';
    cropControls.style.display = 'none';
    cropSelection = null;
  }
}

function startCrop(e) {
  if (!isCropMode || !previewImage.classList.contains('loaded')) return;

  const rect = imageContainer.getBoundingClientRect();
  cropStartX = e.clientX - rect.left;
  cropStartY = e.clientY - rect.top;
  isCropping = true;

  selectionOverlay.style.display = 'block';
  selectionOverlay.style.left = cropStartX + 'px';
  selectionOverlay.style.top = cropStartY + 'px';
  selectionOverlay.style.width = '0px';
  selectionOverlay.style.height = '0px';
}

function updateCrop(e) {
  if (!isCropping) return;

  const rect = imageContainer.getBoundingClientRect();
  const currentX = Math.max(0, Math.min(e.clientX - rect.left, rect.width));
  const currentY = Math.max(0, Math.min(e.clientY - rect.top, rect.height));

  const left = Math.min(cropStartX, currentX);
  const top = Math.min(cropStartY, currentY);
  const width = Math.abs(currentX - cropStartX);
  const height = Math.abs(currentY - cropStartY);

  selectionOverlay.style.left = left + 'px';
  selectionOverlay.style.top = top + 'px';
  selectionOverlay.style.width = width + 'px';
  selectionOverlay.style.height = height + 'px';
}

function endCrop(e) {
  if (!isCropping) return;
  isCropping = false;

  const rect = imageContainer.getBoundingClientRect();
  const overlay = selectionOverlay.getBoundingClientRect();

  if (overlay.width > 10 && overlay.height > 10) {
    cropSelection = {
      x: (overlay.left - rect.left) / rect.width,
      y: (overlay.top - rect.top) / rect.height,
      width: overlay.width / rect.width,
      height: overlay.height / rect.height
    };
    cropControls.style.display = 'flex';
  } else {
    selectionOverlay.style.display = 'none';
    cropSelection = null;
  }
}

function cancelCrop() {
  selectionOverlay.style.display = 'none';
  cropControls.style.display = 'none';
  cropSelection = null;
}

async function confirmCrop() {
  if (!cropSelection || !currentImageBlob) return;

  const img = new Image();
  img.src = URL.createObjectURL(currentImageBlob);

  await new Promise(resolve => img.onload = resolve);

  const canvas = document.createElement('canvas');
  const ctx = canvas.getContext('2d');

  const sx = cropSelection.x * img.width;
  const sy = cropSelection.y * img.height;
  const sw = cropSelection.width * img.width;
  const sh = cropSelection.height * img.height;

  canvas.width = sw;
  canvas.height = sh;
  ctx.drawImage(img, sx, sy, sw, sh, 0, 0, sw, sh);

  canvas.toBlob((blob) => {
    currentImageBlob = blob;
    previewImage.src = URL.createObjectURL(blob);
    cancelCrop();
    toggleCropMode();
  }, 'image/jpeg', 0.9);
}

// ==================== SAVE TO BATCH ====================

async function handleBatchChange() {
  currentBatch = batchSelect.value || null;
  await chrome.storage.local.set({ currentBatch });
  updateBatchInfo();
  saveToBatchBtn.disabled = !currentBatch || !currentImageBlob;
}

async function saveCurrentImageToBatch() {
  if (!currentBatch || !currentImageBlob) return;

  const settings = await chrome.storage.local.get(['optimize_images', 'image_quality', 'max_dimension']);
  let blobToSave = currentImageBlob;

  if (settings.optimize_images !== false) {
    blobToSave = await optimizeImage(currentImageBlob, settings);
  }

  // Convert blob to base64 for storage
  const base64 = await blobToBase64(blobToSave);

  const imageData = {
    batch: currentBatch,
    timestamp: new Date().toISOString(),
    params: { ...currentParams },
    image: base64,
    size: blobToSave.size
  };

  await saveImageToDB(imageData);
  await updateBatchInfo();

  // Visual feedback
  saveToBatchBtn.textContent = 'Saved!';
  saveToBatchBtn.style.backgroundColor = '#137333';
  setTimeout(() => {
    saveToBatchBtn.textContent = 'Save to Batch';
    saveToBatchBtn.style.backgroundColor = '';
  }, 1000);
}

async function optimizeImage(blob, settings) {
  const maxDim = settings.max_dimension || 1024;
  const quality = settings.image_quality || 0.8;

  const img = new Image();
  img.src = URL.createObjectURL(blob);
  await new Promise(resolve => img.onload = resolve);

  let { width, height } = img;
  if (width > maxDim || height > maxDim) {
    if (width > height) {
      height = (height / width) * maxDim;
      width = maxDim;
    } else {
      width = (width / height) * maxDim;
      height = maxDim;
    }
  }

  const canvas = document.createElement('canvas');
  canvas.width = width;
  canvas.height = height;
  const ctx = canvas.getContext('2d');
  ctx.drawImage(img, 0, 0, width, height);

  return new Promise(resolve => {
    canvas.toBlob(resolve, 'image/jpeg', quality);
  });
}

function blobToBase64(blob) {
  return new Promise((resolve, reject) => {
    const reader = new FileReader();
    reader.onloadend = () => resolve(reader.result);
    reader.onerror = reject;
    reader.readAsDataURL(blob);
  });
}

function base64ToBlob(base64) {
  const parts = base64.split(';base64,');
  const contentType = parts[0].split(':')[1];
  const raw = window.atob(parts[1]);
  const rawLength = raw.length;
  const uInt8Array = new Uint8Array(rawLength);
  for (let i = 0; i < rawLength; ++i) {
    uInt8Array[i] = raw.charCodeAt(i);
  }
  return new Blob([uInt8Array], { type: contentType });
}

// ==================== GALLERY ====================

async function renderGallery() {
  const batchFilter = galleryBatchFilter.value;
  const images = await getImagesByBatch(batchFilter);

  // Update button states
  uploadBatchBtn.disabled = images.length === 0;
  deleteBatchBtn.disabled = batchFilter === 'all' || images.length === 0;

  // Calculate stats
  const totalSize = images.reduce((sum, img) => sum + (img.size || 0), 0);
  galleryCount.textContent = `${images.length} image${images.length !== 1 ? 's' : ''}`;
  gallerySize.textContent = formatBytes(totalSize);

  // Render grid
  if (images.length === 0) {
    galleryGrid.innerHTML = '<p class="empty-message">No images in this batch</p>';
    return;
  }

  galleryGrid.innerHTML = images.map(img => `
    <div class="gallery-item" data-id="${img.id}">
      <img src="${img.image}" alt="Captured image">
      <span class="batch-label">${img.batch}</span>
    </div>
  `).join('');

  // Add click handlers
  galleryGrid.querySelectorAll('.gallery-item').forEach(item => {
    item.addEventListener('click', () => openImageModal(parseInt(item.dataset.id)));
  });
}

async function openImageModal(imageId) {
  const images = await getAllImages();
  const image = images.find(img => img.id === imageId);
  if (!image) return;

  modalImage.src = image.image;
  modalDeleteBtn.dataset.id = imageId;
  showModal(imageModal);
}

async function deleteModalImage() {
  const id = parseInt(modalDeleteBtn.dataset.id);
  await deleteImageFromDB(id);
  hideModal(imageModal);
  await renderGallery();
  await updateBatchInfo();
}

function formatBytes(bytes) {
  if (bytes === 0) return '0 B';
  const k = 1024;
  const sizes = ['B', 'KB', 'MB', 'GB'];
  const i = Math.floor(Math.log(bytes) / Math.log(k));
  return parseFloat((bytes / Math.pow(k, i)).toFixed(1)) + ' ' + sizes[i];
}

// ==================== ROBOFLOW UPLOAD ====================

async function uploadToRoboflow() {
  const settings = await chrome.storage.local.get(['roboflow_api_key', 'roboflow_project']);

  if (!settings.roboflow_api_key || !settings.roboflow_project) {
    alert('Please configure your Roboflow API key and project in Settings.');
    return;
  }

  const batchFilter = galleryBatchFilter.value;
  const images = await getImagesByBatch(batchFilter);

  if (images.length === 0) {
    alert('No images to upload.');
    return;
  }

  const batchName = batchFilter === 'all' ? 'traffic-assistant' : batchFilter;

  uploadProgress.style.display = 'block';
  uploadBatchBtn.disabled = true;
  progressFill.style.width = '0%';
  progressText.textContent = `Uploading 0/${images.length}...`;

  let uploaded = 0;
  let errors = 0;

  for (const img of images) {
    try {
      const blob = base64ToBlob(img.image);
      const filename = `img_${img.id}_${Date.now()}.jpg`;

      const uploadUrl = `https://api.roboflow.com/dataset/${settings.roboflow_project}/upload?api_key=${settings.roboflow_api_key}&batch=${encodeURIComponent(batchName)}&name=${encodeURIComponent(filename)}`;

      const formData = new FormData();
      formData.append('file', blob, filename);

      const response = await fetch(uploadUrl, {
        method: 'POST',
        body: formData
      });

      if (response.ok) {
        uploaded++;
      } else {
        errors++;
        console.error('Upload failed:', await response.text());
      }
    } catch (error) {
      errors++;
      console.error('Upload error:', error);
    }

    const progress = ((uploaded + errors) / images.length) * 100;
    progressFill.style.width = progress + '%';
    progressText.textContent = `Uploaded ${uploaded}/${images.length}${errors > 0 ? ` (${errors} errors)` : ''}...`;
  }

  progressText.textContent = `Complete! ${uploaded} uploaded${errors > 0 ? `, ${errors} failed` : ''}`;
  uploadBatchBtn.disabled = false;

  setTimeout(() => {
    uploadProgress.style.display = 'none';
  }, 3000);
}

// ==================== SETTINGS HELPERS ====================

function handleRoboflowUrlPaste(e) {
  const url = e.target.value || '';
  // Parse URLs like: https://app.roboflow.com/workspace-name/project-name/...
  const match = url.match(/app\.roboflow\.com\/([^/]+)\/([^/]+)/);
  if (match) {
    roboflowWorkspace.value = match[1];
    roboflowProject.value = match[2];
    roboflowUrlPaste.value = '';
    showSettingsStatus('Workspace and project extracted from URL', 'success');
  }
}

async function exportAllData() {
  const images = await getAllImages();
  const settings = await chrome.storage.local.get(null);

  const exportData = {
    exportDate: new Date().toISOString(),
    settings: {
      batches: settings.batches,
      roboflow_project: settings.roboflow_project,
      roboflow_workspace: settings.roboflow_workspace
    },
    images: images.map(img => ({
      ...img,
      image: '[BASE64_DATA]' // Don't include actual image data in JSON export
    }))
  };

  const blob = new Blob([JSON.stringify(exportData, null, 2)], { type: 'application/json' });
  const url = URL.createObjectURL(blob);
  const a = document.createElement('a');
  a.href = url;
  a.download = `traffic-assistant-export-${Date.now()}.json`;
  a.click();
}

async function clearAllData() {
  if (!confirm('This will delete ALL saved images and batches. Are you sure?')) return;
  if (!confirm('This cannot be undone. Really delete everything?')) return;

  await clearAllImages();
  await chrome.storage.local.set({ batches: [], currentBatch: null });
  await loadBatches();
  await renderGallery();
  currentBatch = null;
  batchSelect.value = '';
  updateBatchInfo();

  showSettingsStatus('All data cleared', 'success');
}

// ==================== TMP DATA TAB ====================

// TMP DOM Elements
const tmpAddress = document.getElementById('tmp-address');
const tmpRoadName = document.getElementById('tmp-road-name');
const tmpSuburb = document.getElementById('tmp-suburb');
const tmpCity = document.getElementById('tmp-city');
const tmpCoords = document.getElementById('tmp-coords');
const tmpWorkType = document.getElementById('tmp-work-type');
const tmpWorkDescription = document.getElementById('tmp-work-description');
const tmpStartDate = document.getElementById('tmp-start-date');
const tmpEndDate = document.getElementById('tmp-end-date');
const tmpShift = document.getElementById('tmp-shift');
const tmpSpeedLimit = document.getElementById('tmp-speed-limit');
const tmpLanes = document.getElementById('tmp-lanes');
const tmpRoadLevel = document.getElementById('tmp-road-level');
const tmpHazardNotes = document.getElementById('tmp-hazard-notes');
const tmpSessionsList = document.getElementById('tmp-sessions-list');

// TMP State
let currentTmpData = null;

// Initialize TMP tab
function initTmpTab() {
  // Set default dates
  const today = new Date().toISOString().split('T')[0];
  tmpStartDate.value = today;
  
  // Event listeners
  document.getElementById('tmp-save-btn')?.addEventListener('click', saveTmpData);
  document.getElementById('tmp-export-btn')?.addEventListener('click', exportTmpData);
  document.getElementById('tmp-clear-btn')?.addEventListener('click', clearTmpForm);
  
  // Load saved sessions
  loadTmpSessions();
}

// Use current Street View location
function useCurentLocationForTmp() {
  if (currentParams && currentParams.lat && currentParams.lng) {
    tmpCoords.textContent = `${currentParams.lat.toFixed(6)}, ${currentParams.lng.toFixed(6)}`;
    
    // Try reverse geocoding (simple approach - just store coords for now)
    // In production, would use Google Geocoding API
    currentTmpData = currentTmpData || {};
    currentTmpData.coordinates = {
      lat: currentParams.lat,
      lng: currentParams.lng,
      heading: currentParams.heading,
      pitch: currentParams.pitch
    };
    
    // Visual feedback
    const btn = document.getElementById('tmp-use-location-btn');
    btn.textContent = '✓ Location Set';
    btn.style.backgroundColor = '#137333';
    btn.style.color = 'white';
    setTimeout(() => {
      btn.textContent = '📍 Use Current Location';
      btn.style.backgroundColor = '';
      btn.style.color = '';
    }, 1500);
  } else {
    alert('No Street View location available. Navigate to a location in Google Maps Street View first.');
  }
}

// Collect all form data
function collectTmpFormData() {
  const docType = document.querySelector('input[name="doc-type"]:checked')?.value || 'COPTTM';
  
  return {
    document_type: docType,
    location: {
      address: tmpAddress.value,
      road_name: tmpRoadName.value,
      suburb: tmpSuburb.value,
      city: tmpCity.value,
      coordinates: currentTmpData?.coordinates || null
    },
    work: {
      type: tmpWorkType.value,
      description: tmpWorkDescription.value,
      start_date: tmpStartDate.value,
      end_date: tmpEndDate.value,
      shift: tmpShift.value
    },
    road: {
      speed_limit: tmpSpeedLimit.value ? parseInt(tmpSpeedLimit.value) : null,
      lanes: tmpLanes.value ? parseInt(tmpLanes.value) : null,
      road_level: tmpRoadLevel.value ? parseInt(tmpRoadLevel.value) : null,
      has_footpath: document.getElementById('tmp-has-footpath').checked,
      has_cycle_lane: document.getElementById('tmp-has-cycle-lane').checked,
      has_parking: document.getElementById('tmp-has-parking').checked,
      has_bus_stop: document.getElementById('tmp-has-bus-stop').checked,
      has_intersection: document.getElementById('tmp-has-intersection').checked,
      has_driveways: document.getElementById('tmp-has-driveways').checked
    },
    aspects_affected: {
      lanes: document.getElementById('tmp-affects-lanes').checked,
      footpath: document.getElementById('tmp-affects-footpath').checked,
      pedestrians: document.getElementById('tmp-affects-pedestrians').checked,
      cycle_lane: document.getElementById('tmp-affects-cycle').checked,
      parking: document.getElementById('tmp-affects-parking').checked,
      intersection: document.getElementById('tmp-affects-intersection').checked,
      bus_stop: document.getElementById('tmp-affects-bus').checked,
      traffic_signals: document.getElementById('tmp-affects-signals').checked
    },
    hazards: {
      curved_road: document.getElementById('tmp-hazard-curve').checked,
      crest_hill: document.getElementById('tmp-hazard-crest').checked,
      narrow_road: document.getElementById('tmp-hazard-narrow').checked,
      school_zone: document.getElementById('tmp-hazard-school').checked,
      hospital_nearby: document.getElementById('tmp-hazard-hospital').checked,
      high_traffic: document.getElementById('tmp-hazard-high-traffic').checked,
      notes: tmpHazardNotes.value
    },
    // Include raw API data for document generation
    api_data: tmpApiData || null,
    captured_at: new Date().toISOString(),
    images: [] // Will be populated with batch images if selected
  };
}

// Save TMP data to storage
async function saveTmpData() {
  const data = collectTmpFormData();
  
  // Validate required fields
  if (!data.location.address && !data.location.road_name) {
    alert('Please enter at least an address or road name.');
    return;
  }
  
  // Generate session name
  const sessionName = data.location.address || data.location.road_name || `TMP-${Date.now()}`;
  data.session_name = sessionName;
  
  // Include images from current batch if any
  if (currentBatch) {
    const images = await getImagesByBatch(currentBatch);
    data.images = images.map(img => ({
      id: img.id,
      timestamp: img.timestamp,
      params: img.params
      // Note: not including full image data to keep storage manageable
    }));
    data.batch_name = currentBatch;
  }
  
  // Load existing sessions
  const result = await chrome.storage.local.get(['tmp_sessions']);
  const sessions = result.tmp_sessions || [];
  
  // Add new session
  sessions.unshift({
    id: Date.now(),
    name: sessionName,
    created_at: new Date().toISOString(),
    data: data
  });
  
  // Keep only last 50 sessions
  if (sessions.length > 50) sessions.length = 50;
  
  await chrome.storage.local.set({ tmp_sessions: sessions });
  
  // Visual feedback
  const btn = document.getElementById('tmp-save-btn');
  btn.textContent = '✓ Saved!';
  btn.style.backgroundColor = '#137333';
  setTimeout(() => {
    btn.textContent = '💾 Save TMP Data';
    btn.style.backgroundColor = '';
  }, 1500);
  
  // Refresh sessions list
  loadTmpSessions();
}

// Export TMP data as JSON
function exportTmpData() {
  const data = collectTmpFormData();
  
  // Create filename
  const location = data.location.address || data.location.road_name || 'unknown';
  const date = new Date().toISOString().split('T')[0];
  const filename = `TMP-${location.replace(/[^a-z0-9]/gi, '-')}-${date}.json`;
  
  // Download
  const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
  const url = URL.createObjectURL(blob);
  const a = document.createElement('a');
  a.href = url;
  a.download = filename;
  a.click();
  URL.revokeObjectURL(url);
}

// Clear TMP form
function clearTmpForm() {
  if (!confirm('Clear all TMP form data?')) return;
  
  // Reset all inputs
  document.querySelectorAll('#tab-tmp input[type="text"], #tab-tmp input[type="date"], #tab-tmp input[type="tel"], #tab-tmp input[type="number"], #tab-tmp textarea').forEach(el => el.value = '');
  document.querySelectorAll('#tab-tmp input[type="checkbox"]').forEach(el => el.checked = false);
  document.querySelectorAll('#tab-tmp select').forEach(el => el.selectedIndex = 0);
  document.querySelector('input[name="doc-type"][value="COPTTM"]').checked = true;
  
  currentTmpData = null;
  tmpApiData = null;
  
  // Hide fetched data section
  const fetchedDataSection = document.getElementById('fetched-data-section');
  if (fetchedDataSection) fetchedDataSection.style.display = 'none';
  
  // Reset dates
  tmpStartDate.value = new Date().toISOString().split('T')[0];
}

// Load saved TMP sessions
async function loadTmpSessions() {
  const result = await chrome.storage.local.get(['tmp_sessions']);
  const sessions = result.tmp_sessions || [];
  
  if (sessions.length === 0) {
    tmpSessionsList.innerHTML = '<p class="empty-message">No saved sessions</p>';
    return;
  }
  
  tmpSessionsList.innerHTML = sessions.map(session => `
    <div class="session-item" data-id="${session.id}">
      <div class="session-info">
        <div class="session-name">${escapeHtml(session.name)}</div>
        <div class="session-date">${new Date(session.created_at).toLocaleDateString()}</div>
      </div>
      <div class="session-actions">
        <button class="btn btn-small btn-icon session-load" data-id="${session.id}" title="Load">📂</button>
        <button class="btn btn-small btn-icon session-export" data-id="${session.id}" title="Export">📤</button>
        <button class="btn btn-small btn-icon btn-danger session-delete" data-id="${session.id}" title="Delete">🗑️</button>
      </div>
    </div>
  `).join('');
  
  // Add event listeners
  tmpSessionsList.querySelectorAll('.session-load').forEach(btn => {
    btn.addEventListener('click', (e) => {
      e.stopPropagation();
      loadTmpSession(parseInt(btn.dataset.id));
    });
  });
  
  tmpSessionsList.querySelectorAll('.session-export').forEach(btn => {
    btn.addEventListener('click', (e) => {
      e.stopPropagation();
      exportTmpSession(parseInt(btn.dataset.id));
    });
  });
  
  tmpSessionsList.querySelectorAll('.session-delete').forEach(btn => {
    btn.addEventListener('click', (e) => {
      e.stopPropagation();
      deleteTmpSession(parseInt(btn.dataset.id));
    });
  });
}

// Load a saved TMP session into the form
async function loadTmpSession(sessionId) {
  const result = await chrome.storage.local.get(['tmp_sessions']);
  const sessions = result.tmp_sessions || [];
  const session = sessions.find(s => s.id === sessionId);
  
  if (!session) return;
  
  const data = session.data;
  
  // Populate form
  document.querySelector(`input[name="doc-type"][value="${data.document_type}"]`).checked = true;
  
  tmpAddress.value = data.location?.address || '';
  tmpRoadName.value = data.location?.road_name || '';
  tmpSuburb.value = data.location?.suburb || '';
  tmpCity.value = data.location?.city || '';
  
  if (data.location?.coordinates) {
    currentTmpData = { coordinates: data.location.coordinates };
  }
  
  tmpWorkType.value = data.work?.type || '';
  tmpWorkDescription.value = data.work?.description || '';
  tmpStartDate.value = data.work?.start_date || '';
  tmpEndDate.value = data.work?.end_date || '';
  tmpShift.value = data.work?.shift || 'day';
  
  tmpSpeedLimit.value = data.road?.speed_limit || '';
  tmpLanes.value = data.road?.lanes || '';
  tmpRoadLevel.value = data.road?.road_level || '';
  
  document.getElementById('tmp-has-footpath').checked = data.road?.has_footpath || false;
  document.getElementById('tmp-has-cycle-lane').checked = data.road?.has_cycle_lane || false;
  document.getElementById('tmp-has-parking').checked = data.road?.has_parking || false;
  document.getElementById('tmp-has-bus-stop').checked = data.road?.has_bus_stop || false;
  document.getElementById('tmp-has-intersection').checked = data.road?.has_intersection || false;
  document.getElementById('tmp-has-driveways').checked = data.road?.has_driveways || false;
  
  document.getElementById('tmp-affects-lanes').checked = data.aspects_affected?.lanes || false;
  document.getElementById('tmp-affects-footpath').checked = data.aspects_affected?.footpath || false;
  document.getElementById('tmp-affects-pedestrians').checked = data.aspects_affected?.pedestrians || false;
  document.getElementById('tmp-affects-cycle').checked = data.aspects_affected?.cycle_lane || false;
  document.getElementById('tmp-affects-parking').checked = data.aspects_affected?.parking || false;
  document.getElementById('tmp-affects-intersection').checked = data.aspects_affected?.intersection || false;
  document.getElementById('tmp-affects-bus').checked = data.aspects_affected?.bus_stop || false;
  document.getElementById('tmp-affects-signals').checked = data.aspects_affected?.traffic_signals || false;
  
  document.getElementById('tmp-hazard-curve').checked = data.hazards?.curved_road || false;
  document.getElementById('tmp-hazard-crest').checked = data.hazards?.crest_hill || false;
  document.getElementById('tmp-hazard-narrow').checked = data.hazards?.narrow_road || false;
  document.getElementById('tmp-hazard-school').checked = data.hazards?.school_zone || false;
  document.getElementById('tmp-hazard-hospital').checked = data.hazards?.hospital_nearby || false;
  document.getElementById('tmp-hazard-high-traffic').checked = data.hazards?.high_traffic || false;
  tmpHazardNotes.value = data.hazards?.notes || '';
  
  // Restore API data if available
  if (data.api_data) {
    tmpApiData = data.api_data;
    displayFetchedData(tmpApiData);
  }
}

// Export a specific session
async function exportTmpSession(sessionId) {
  const result = await chrome.storage.local.get(['tmp_sessions']);
  const sessions = result.tmp_sessions || [];
  const session = sessions.find(s => s.id === sessionId);
  
  if (!session) return;
  
  const filename = `TMP-${session.name.replace(/[^a-z0-9]/gi, '-')}.json`;
  const blob = new Blob([JSON.stringify(session.data, null, 2)], { type: 'application/json' });
  const url = URL.createObjectURL(blob);
  const a = document.createElement('a');
  a.href = url;
  a.download = filename;
  a.click();
  URL.revokeObjectURL(url);
}

// Delete a session
async function deleteTmpSession(sessionId) {
  if (!confirm('Delete this session?')) return;
  
  const result = await chrome.storage.local.get(['tmp_sessions']);
  let sessions = result.tmp_sessions || [];
  sessions = sessions.filter(s => s.id !== sessionId);
  
  await chrome.storage.local.set({ tmp_sessions: sessions });
  loadTmpSessions();
}

// Helper to escape HTML
function escapeHtml(text) {
  const div = document.createElement('div');
  div.textContent = text;
  return div.innerHTML;
}

// Initialize TMP tab when DOM ready
document.addEventListener('DOMContentLoaded', initTmpTab);

// ==================== WORD DOC GENERATION ====================

// Hazard/control mappings (from TMP-analysis)
const HAZARD_CONTROLS = {
  "traffic_during_setup": {
    hazard: "Traffic",
    risk_type: "Implementation/Removal Hazard",
    description: "TTM workers struck by vehicles during setup or removal of TTM equipment causing injury or fatality",
    controls: ["High-vis PPE at all times", "Advance warning signs deployed first", "Work facing oncoming traffic", "Spotter/lookout during setup"]
  },
  "manual_handling": {
    hazard: "TTM Equipment",
    risk_type: "Implementation/Removal Hazard", 
    description: "Manual handling of items causing sprains, strains, or injury from over lifting",
    controls: ["Use mechanical aids where possible", "Team lifting for heavy items", "Training in safe lifting techniques"]
  },
  "worker_struck_during_activity": {
    hazard: "Traffic",
    risk_type: "Activity Hazard",
    description: "Risk of workers being struck by passing traffic during work activity",
    controls: ["Adequate safety zone separation", "Delineation devices (cones, barriers)", "TSL to reduce vehicle speeds", "STMS supervision"]
  },
  "vulnerable_road_users": {
    hazard: "Vulnerable Road Users",
    risk_type: "Activity Hazard",
    description: "Pedestrians, cyclists, or mobility impaired users impacted by work site",
    controls: ["Safe pedestrian detour route", "Clear signage for alternate routes", "Cyclist warning signs"]
  },
  "communication_failure": {
    hazard: "Communication",
    risk_type: "Implementation/Removal Hazard",
    description: "Lack of communication between staff leading to unsafe practices",
    controls: ["Toolbox meeting before work", "Radio communication between staff", "Clear chain of command"]
  }
};

const WORK_TYPE_HAZARDS = {
  "excavation": ["traffic_during_setup", "manual_handling", "worker_struck_during_activity", "vulnerable_road_users"],
  "scaffolding": ["traffic_during_setup", "manual_handling", "worker_struck_during_activity", "vulnerable_road_users"],
  "cctv": ["traffic_during_setup", "worker_struck_during_activity", "vulnerable_road_users"],
  "fiber": ["traffic_during_setup", "manual_handling", "worker_struck_during_activity", "vulnerable_road_users"],
  "drainage": ["traffic_during_setup", "manual_handling", "worker_struck_during_activity", "vulnerable_road_users"],
  "maintenance": ["traffic_during_setup", "worker_struck_during_activity", "manual_handling"],
  "construction": ["traffic_during_setup", "manual_handling", "worker_struck_during_activity", "vulnerable_road_users", "communication_failure"],
  "event": ["traffic_during_setup", "vulnerable_road_users", "communication_failure"]
};

function getHazardsForWorkType(workType) {
  const hazardKeys = WORK_TYPE_HAZARDS[workType] || WORK_TYPE_HAZARDS["maintenance"];
  return hazardKeys.map(key => HAZARD_CONTROLS[key]).filter(Boolean);
}

async function generateWordDoc() {
  const data = collectTmpFormData();
  
  if (!data.location.address && !data.location.road_name) {
    alert('Please enter at least an address or road name.');
    return;
  }
  
  const btn = document.getElementById('tmp-generate-doc-btn');
  btn.textContent = '⏳ Generating...';
  btn.disabled = true;
  
  try {
    const response = await fetch('https://traffic-agent.fly.dev/generate', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(data)
    });
    
    if (!response.ok) {
      const error = await response.json();
      throw new Error(error.error || 'Generation failed');
    }
    
    // Download the file
    const blob = await response.blob();
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    
    const location = data.location.address || data.location.road_name || 'TMP';
    a.download = `TMP-${location.replace(/[^a-z0-9]/gi, '-')}.docx`;
    a.click();
    URL.revokeObjectURL(url);
    
    btn.textContent = '✓ Downloaded!';
    btn.style.backgroundColor = '#137333';
    setTimeout(() => {
      btn.textContent = '📄 Generate Word Doc';
      btn.style.backgroundColor = '';
      btn.disabled = false;
    }, 2000);
    
  } catch (error) {
    console.error('Generation error:', error);
    alert('Failed to generate document: ' + error.message);
    btn.textContent = '📄 Generate Word Doc';
    btn.disabled = false;
  }
}

// Add event listener for generate button
document.getElementById('tmp-generate-doc-btn')?.addEventListener('click', generateWordDoc);

// Generate HTML/PDF version
async function generateHTML() {
  const data = collectTmpFormData();
  
  if (!data.location.address && !data.location.road_name) {
    alert('Please enter at least an address or road name.');
    return;
  }
  
  const btn = document.getElementById('tmp-generate-html-btn');
  btn.textContent = '⏳ Generating...';
  btn.disabled = true;
  
  try {
    const response = await fetch('https://traffic-agent.fly.dev/generate/html', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(data)
    });
    
    if (!response.ok) {
      const error = await response.json();
      throw new Error(error.error || 'Generation failed');
    }
    
    // Get the HTML and open in new tab for preview/print
    const html = await response.text();
    const blob = new Blob([html], { type: 'text/html' });
    const url = URL.createObjectURL(blob);
    
    // Open in new tab - user can then print to PDF
    window.open(url, '_blank');
    
    btn.textContent = '✓ Opened!';
    btn.style.backgroundColor = '#137333';
    setTimeout(() => {
      btn.textContent = '🌐 HTML/PDF';
      btn.style.backgroundColor = '';
      btn.disabled = false;
    }, 2000);
    
  } catch (error) {
    console.error('Generation error:', error);
    alert('Failed to generate: ' + error.message);
    btn.textContent = '🌐 HTML/PDF';
    btn.disabled = false;
  }
}

// Add event listener for HTML generate button
document.getElementById('tmp-generate-html-btn')?.addEventListener('click', generateHTML);

// ==================== DATA SOURCES TAB ====================

// Default bookmarks from Dan's list
const DEFAULT_BOOKMARKS = [
  // National
  { id: "google-maps", name: "Google Maps", url: "https://www.google.com/maps", category: "national", region: null, icon: "🗺️", urlTemplate: "https://www.google.com/maps/@{lat},{lng},17z" },
  { id: "mobile-road", name: "Mobile Road", url: "https://www.mobileroad.org/", category: "national", region: null, icon: "🚗", description: "Traffic counts & road details" },
  { id: "nslr-speed", name: "NSLR Speed Limits", url: "https://speedlimits.nzta.govt.nz/", category: "national", region: null, icon: "🚦", description: "All NZ speed limits" },
  { id: "onrc", name: "One Network Classification", url: "https://nzta.maps.arcgis.com/apps/webappviewer/index.html?id=95fad5204ad243c39d84c37701f614b0", category: "national", region: null, icon: "🛣️" },
  { id: "nzta-detours", name: "NZTA Detour Selector", url: "https://detours.myworksites.co.nz/", category: "national", region: null, icon: "↩️", description: "State Highway closures" },
  
  // Canterbury/Christchurch
  { id: "canterbury-maps", name: "Canterbury Maps", url: "https://mapviewer.canterburymaps.govt.nz/", category: "regional", region: "canterbury", icon: "🗺️", description: "CCC detailed map" },
  { id: "ccc-traffic-links", name: "CCC Links Traffic", url: "https://ccc.govt.nz/transport/improving-our-transport-and-roads/traffic-count-data/links-traffic-counts-dashboard", category: "regional", region: "canterbury", icon: "📊", description: "Hourly traffic data" },
  { id: "ccc-traffic-intersections", name: "CCC Intersection Traffic", url: "https://ccc.govt.nz/transport/improving-our-transport-and-roads/traffic-count-data/intersection-traffic-counts-database", category: "regional", region: "canterbury", icon: "📊" },
  { id: "ccc-road-levels", name: "CCC Road Levels", url: "https://ccc.govt.nz/assets/Documents/Transport/Working-on-our-roads/TMP/CCC-Road-Levels-Booklet-.pdf", category: "regional", region: "canterbury", icon: "📄" },
  { id: "ccc-3waters", name: "CCC 3-Waters", url: "https://gis.ccc.govt.nz/portal/apps/webappviewer/index.html?id=c4ff85ba75724391ae89fbf5f773e00d", category: "regional", region: "canterbury", icon: "💧" },
  { id: "metro-chch", name: "Metro Christchurch", url: "https://www.metroinfo.co.nz/", category: "regional", region: "canterbury", icon: "🚌", description: "Bus routes" },
  
  // Wellington
  { id: "wcc-service-locator", name: "WCC Service Locator", url: "https://wcc.maps.arcgis.com/apps/webappviewer/index.html?id=7b0a8513d0f643d3a77f82918e8dbf58", category: "regional", region: "wellington", icon: "🗺️" },
  { id: "wcc-vehicle-counts", name: "Wellington Vehicle Counts", url: "https://wellington.govt.nz/parking-roads-and-transport/roads/working-on-the-road/vehicle-counts", category: "regional", region: "wellington", icon: "📊" },
  { id: "gwrc-bus", name: "GWRC Bus Routes", url: "https://mapping.gw.govt.nz/GW/GWpublicMap_Mobile/?webmap=584df1cd3d3a4df9af2ea18f41c45fc3", category: "regional", region: "wellington", icon: "🚌" },
  
  // References
  { id: "m23-safety", name: "M23 Safety Hardware", url: "https://www.nzta.govt.nz/assets/resources/road-safety-barrier-systems/docs/m23-road-safety-barrier-systems-appendix-f-working-draft-release.pdf", category: "reference", region: null, icon: "📋", description: "Device specs" },
  { id: "nzta-resources", name: "NZTA Resources", url: "https://nzta.govt.nz/roads-and-rail/accessing-our-state-highways/helpful-resources-and-tools", category: "reference", region: null, icon: "📋", description: "Regional requirements" },
];

// Region bounding boxes (approximate)
const REGIONS = {
  canterbury: { 
    name: "Canterbury", 
    bounds: { latMin: -44.5, latMax: -42.0, lngMin: 170.0, lngMax: 173.5 } 
  },
  wellington: { 
    name: "Wellington", 
    bounds: { latMin: -41.6, latMax: -40.7, lngMin: 174.5, lngMax: 176.2 } 
  },
  auckland: {
    name: "Auckland",
    bounds: { latMin: -37.5, latMax: -36.0, lngMin: 174.0, lngMax: 175.5 }
  },
  otago: {
    name: "Otago",
    bounds: { latMin: -46.5, latMax: -44.0, lngMin: 168.0, lngMax: 171.5 }
  }
};

// Sources state
let sourcesState = {
  bookmarks: [],
  userBookmarks: [],
  hiddenDefaults: [],
  detectedRegion: null
};

// DOM elements
const sourcesSearch = document.getElementById('sources-search');
const sourcesRegionSection = document.getElementById('sources-region-section');
const sourcesRegionName = document.getElementById('sources-region-name');
const sourcesRegionList = document.getElementById('sources-region-list');
const sourcesNationalList = document.getElementById('sources-national-list');
const sourcesReferenceList = document.getElementById('sources-reference-list');
const sourcesUserList = document.getElementById('sources-user-list');
const addBookmarkModal = document.getElementById('add-bookmark-modal');

// Initialize sources tab
async function initSourcesTab() {
  await loadSourcesPrefs();
  renderAllSources();
  setupSourcesListeners();
}

// Load user preferences
async function loadSourcesPrefs() {
  const result = await chrome.storage.sync.get(['sourcesPrefs']);
  const prefs = result.sourcesPrefs || {};
  
  sourcesState.userBookmarks = prefs.userBookmarks || [];
  sourcesState.hiddenDefaults = prefs.hiddenDefaults || [];
}

// Save user preferences
async function saveSourcesPrefs() {
  await chrome.storage.sync.set({
    sourcesPrefs: {
      userBookmarks: sourcesState.userBookmarks,
      hiddenDefaults: sourcesState.hiddenDefaults
    }
  });
}

// Detect region from coordinates
function detectRegion(lat, lng) {
  if (!lat || !lng) return null;
  
  for (const [regionId, region] of Object.entries(REGIONS)) {
    const b = region.bounds;
    if (lat >= b.latMin && lat <= b.latMax && lng >= b.lngMin && lng <= b.lngMax) {
      return { id: regionId, name: region.name };
    }
  }
  return null;
}

// Update detected region based on current params
function updateDetectedRegion() {
  if (currentParams && currentParams.lat && currentParams.lng) {
    sourcesState.detectedRegion = detectRegion(currentParams.lat, currentParams.lng);
  } else {
    sourcesState.detectedRegion = null;
  }
  renderAllSources();
}

// Build URL with coordinates if template available
function buildSourceUrl(bookmark) {
  if (bookmark.urlTemplate && currentParams && currentParams.lat && currentParams.lng) {
    return bookmark.urlTemplate
      .replace('{lat}', currentParams.lat)
      .replace('{lng}', currentParams.lng);
  }
  return bookmark.url;
}

// Render a single source item
function renderSourceItem(bookmark, isUser = false) {
  const isHidden = sourcesState.hiddenDefaults.includes(bookmark.id);
  if (isHidden && !isUser) return '';
  
  const url = buildSourceUrl(bookmark);
  
  return `
    <div class="source-item" data-id="${bookmark.id}" data-url="${url}">
      <span class="source-icon">${bookmark.icon || '🔗'}</span>
      <div class="source-info">
        <div class="source-name">${escapeHtml(bookmark.name)}</div>
        <div class="source-url">${escapeHtml(bookmark.description || new URL(bookmark.url).hostname)}</div>
      </div>
      <div class="source-actions">
        ${isUser ? `<button class="delete" data-id="${bookmark.id}" title="Remove">✕</button>` : ''}
      </div>
    </div>
  `;
}

// Render all sources
function renderAllSources() {
  const searchTerm = (sourcesSearch?.value || '').toLowerCase();
  
  // Filter function
  const matchesSearch = (b) => !searchTerm || 
    b.name.toLowerCase().includes(searchTerm) || 
    (b.description && b.description.toLowerCase().includes(searchTerm));
  
  // Regional sources
  if (sourcesState.detectedRegion) {
    sourcesRegionSection.classList.remove('no-region');
    sourcesRegionName.textContent = sourcesState.detectedRegion.name;
    
    const regionalBookmarks = DEFAULT_BOOKMARKS.filter(b => 
      b.region === sourcesState.detectedRegion.id && matchesSearch(b)
    );
    sourcesRegionList.innerHTML = regionalBookmarks.map(b => renderSourceItem(b)).join('') 
      || '<p class="empty-message">No sources for this region</p>';
  } else {
    sourcesRegionSection.classList.add('no-region');
  }
  
  // National sources
  const nationalBookmarks = DEFAULT_BOOKMARKS.filter(b => 
    b.category === 'national' && matchesSearch(b)
  );
  sourcesNationalList.innerHTML = nationalBookmarks.map(b => renderSourceItem(b)).join('');
  
  // Reference sources
  const referenceBookmarks = DEFAULT_BOOKMARKS.filter(b => 
    b.category === 'reference' && matchesSearch(b)
  );
  sourcesReferenceList.innerHTML = referenceBookmarks.map(b => renderSourceItem(b)).join('');
  
  // User bookmarks
  const userBookmarks = sourcesState.userBookmarks.filter(matchesSearch);
  if (userBookmarks.length > 0) {
    sourcesUserList.innerHTML = userBookmarks.map(b => renderSourceItem(b, true)).join('');
  } else {
    sourcesUserList.innerHTML = '<p class="empty-message">No custom bookmarks</p>';
  }
  
  // Add click handlers
  document.querySelectorAll('.source-item').forEach(item => {
    item.addEventListener('click', (e) => {
      if (e.target.closest('.source-actions')) return;
      const url = item.dataset.url;
      chrome.tabs.create({ url });
    });
  });
  
  // Add delete handlers for user bookmarks
  document.querySelectorAll('.source-actions .delete').forEach(btn => {
    btn.addEventListener('click', (e) => {
      e.stopPropagation();
      deleteUserBookmark(btn.dataset.id);
    });
  });
}

// Setup event listeners
function setupSourcesListeners() {
  // Search filter
  sourcesSearch?.addEventListener('input', () => {
    renderAllSources();
  });
  
  // Open all regional sources
  document.getElementById('sources-open-region')?.addEventListener('click', () => {
    if (!sourcesState.detectedRegion) return;
    
    const regionalBookmarks = DEFAULT_BOOKMARKS.filter(b => 
      b.region === sourcesState.detectedRegion.id && 
      !sourcesState.hiddenDefaults.includes(b.id)
    );
    
    regionalBookmarks.forEach(b => {
      chrome.tabs.create({ url: buildSourceUrl(b) });
    });
  });
  
  // Add bookmark button
  document.getElementById('sources-add-bookmark')?.addEventListener('click', () => {
    document.getElementById('bookmark-name').value = '';
    document.getElementById('bookmark-url').value = '';
    addBookmarkModal.style.display = 'flex';
  });
  
  // Cancel add bookmark
  document.getElementById('cancel-bookmark-btn')?.addEventListener('click', () => {
    addBookmarkModal.style.display = 'none';
  });
  
  // Save bookmark
  document.getElementById('save-bookmark-btn')?.addEventListener('click', () => {
    const name = document.getElementById('bookmark-name').value.trim();
    const url = document.getElementById('bookmark-url').value.trim();
    
    if (!name || !url) {
      alert('Please enter both name and URL');
      return;
    }
    
    try {
      new URL(url); // Validate URL
    } catch {
      alert('Please enter a valid URL');
      return;
    }
    
    addUserBookmark(name, url);
    addBookmarkModal.style.display = 'none';
  });
  
  // Close modal on backdrop click
  addBookmarkModal?.addEventListener('click', (e) => {
    if (e.target === addBookmarkModal) {
      addBookmarkModal.style.display = 'none';
    }
  });
  
  // Reset to defaults
  document.getElementById('sources-reset-defaults')?.addEventListener('click', async () => {
    if (!confirm('Reset all sources to defaults? This will remove custom bookmarks.')) return;
    sourcesState.userBookmarks = [];
    sourcesState.hiddenDefaults = [];
    await saveSourcesPrefs();
    renderAllSources();
  });
  
  // Export bookmarks
  document.getElementById('sources-export')?.addEventListener('click', () => {
    const exportData = {
      userBookmarks: sourcesState.userBookmarks,
      hiddenDefaults: sourcesState.hiddenDefaults,
      exportedAt: new Date().toISOString()
    };
    
    const blob = new Blob([JSON.stringify(exportData, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'tmp-sources-backup.json';
    a.click();
    URL.revokeObjectURL(url);
  });
  
  // Import bookmarks
  document.getElementById('sources-import')?.addEventListener('click', () => {
    document.getElementById('sources-import-file').click();
  });
  
  document.getElementById('sources-import-file')?.addEventListener('change', async (e) => {
    const file = e.target.files[0];
    if (!file) return;
    
    try {
      const text = await file.text();
      const data = JSON.parse(text);
      
      if (data.userBookmarks) {
        sourcesState.userBookmarks = data.userBookmarks;
      }
      if (data.hiddenDefaults) {
        sourcesState.hiddenDefaults = data.hiddenDefaults;
      }
      
      await saveSourcesPrefs();
      renderAllSources();
      alert('Bookmarks imported successfully');
    } catch (err) {
      alert('Failed to import: ' + err.message);
    }
    
    e.target.value = ''; // Reset file input
  });
}

// Add user bookmark
async function addUserBookmark(name, url) {
  const bookmark = {
    id: 'user-' + Date.now(),
    name,
    url,
    icon: '⭐',
    category: 'user'
  };
  
  sourcesState.userBookmarks.push(bookmark);
  await saveSourcesPrefs();
  renderAllSources();
}

// Delete user bookmark
async function deleteUserBookmark(id) {
  sourcesState.userBookmarks = sourcesState.userBookmarks.filter(b => b.id !== id);
  await saveSourcesPrefs();
  renderAllSources();
}

// Hook into URL updates to detect region
const originalHandleUrlUpdate = handleUrlUpdate;
handleUrlUpdate = function(url) {
  originalHandleUrlUpdate(url);
  updateDetectedRegion();
};

// Initialize on DOM ready
document.addEventListener('DOMContentLoaded', initSourcesTab);

// ==================== PINNED LOCATION ====================

let pinnedLocation = null;

// DOM elements for pinning
const pinnedLocationDiv = document.getElementById('pinned-location');
const pinnedCoordsSpan = document.getElementById('pinned-coords');
const pinnedTimeSpan = document.getElementById('pinned-time');
const pinnedDistanceDiv = document.getElementById('pinned-distance');
const distanceText = document.getElementById('distance-text');
const pinLocationBtn = document.getElementById('pin-location-btn');
const updatePinBtn = document.getElementById('update-pin-btn');
const clearPinBtn = document.getElementById('clear-pin-btn');
const pinHint = document.getElementById('pin-hint');

// Initialize pinned location system
function initPinnedLocation() {
  pinLocationBtn?.addEventListener('click', pinCurrentLocation);
  updatePinBtn?.addEventListener('click', pinCurrentLocation);
  clearPinBtn?.addEventListener('click', clearPinnedLocation);
  
  // Load saved pin from session storage
  const saved = sessionStorage.getItem('pinnedLocation');
  if (saved) {
    pinnedLocation = JSON.parse(saved);
    updatePinnedLocationUI();
  }
}

// Pin the current Street View location
function pinCurrentLocation() {
  if (!currentParams || !currentParams.lat || !currentParams.lng) {
    alert('Navigate to a Street View location first.');
    return;
  }
  
  pinnedLocation = {
    lat: currentParams.lat,
    lng: currentParams.lng,
    heading: currentParams.heading,
    pitch: currentParams.pitch,
    timestamp: new Date().toISOString()
  };
  
  // Save to session storage
  sessionStorage.setItem('pinnedLocation', JSON.stringify(pinnedLocation));
  
  // Update UI
  updatePinnedLocationUI();
  
  // Update TMP form data
  if (typeof currentTmpData === 'undefined') {
    window.currentTmpData = {};
  }
  currentTmpData.coordinates = pinnedLocation;
  
  // Trigger hazard suggestions
  updateHazardSuggestions();
}

// Clear the pinned location
function clearPinnedLocation() {
  pinnedLocation = null;
  sessionStorage.removeItem('pinnedLocation');
  currentTmpData = currentTmpData || {};
  currentTmpData.coordinates = null;
  
  updatePinnedLocationUI();
}

// Update the pinned location UI
function updatePinnedLocationUI() {
  if (pinnedLocation) {
    pinnedLocationDiv.style.display = 'block';
    pinnedCoordsSpan.textContent = `${pinnedLocation.lat.toFixed(6)}, ${pinnedLocation.lng.toFixed(6)}`;
    
    const time = new Date(pinnedLocation.timestamp);
    pinnedTimeSpan.textContent = `Pinned at ${time.toLocaleTimeString()}`;
    
    pinLocationBtn.style.display = 'none';
    updatePinBtn.style.display = 'inline-flex';
    pinHint.textContent = 'Location pinned. Browse freely - pin stays fixed.';
    
    // Update distance display
    updateDistanceFromPin();
  } else {
    pinnedLocationDiv.style.display = 'none';
    pinLocationBtn.style.display = 'inline-flex';
    updatePinBtn.style.display = 'none';
    pinHint.textContent = 'Pin the work site location, then browse freely to check surroundings.';
    pinnedDistanceDiv.style.display = 'none';
  }
}

// Calculate and display distance from current location to pin
function updateDistanceFromPin() {
  if (!pinnedLocation || !currentParams || !currentParams.lat || !currentParams.lng) {
    pinnedDistanceDiv.style.display = 'none';
    return;
  }
  
  const distance = calculateDistance(
    pinnedLocation.lat, pinnedLocation.lng,
    currentParams.lat, currentParams.lng
  );
  
  if (distance > 10) { // More than 10 meters away
    pinnedDistanceDiv.style.display = 'flex';
    
    if (distance >= 1000) {
      distanceText.textContent = `${(distance / 1000).toFixed(1)} km from pinned location`;
    } else {
      distanceText.textContent = `${Math.round(distance)} m from pinned location`;
    }
    
    // Change color based on distance
    pinnedDistanceDiv.classList.toggle('far', distance > 500);
  } else {
    pinnedDistanceDiv.style.display = 'none';
  }
}

// Haversine formula for distance calculation
function calculateDistance(lat1, lon1, lat2, lon2) {
  const R = 6371000; // Earth's radius in meters
  const dLat = (lat2 - lat1) * Math.PI / 180;
  const dLon = (lon2 - lon1) * Math.PI / 180;
  const a = Math.sin(dLat/2) * Math.sin(dLat/2) +
            Math.cos(lat1 * Math.PI / 180) * Math.cos(lat2 * Math.PI / 180) *
            Math.sin(dLon/2) * Math.sin(dLon/2);
  const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
  return R * c;
}

// Hook into URL updates to track distance
const originalHandleUrlUpdateForPin = handleUrlUpdate;
handleUrlUpdate = function(url) {
  originalHandleUrlUpdateForPin(url);
  updateDistanceFromPin();
};

// ==================== HAZARD SUGGESTIONS ====================

const hazardSuggestionsDiv = document.getElementById('hazard-suggestions');
const suggestionsListDiv = document.getElementById('suggestions-list');

// Hazard suggestion rules
const HAZARD_RULES = [
  {
    id: 'high_speed',
    hazardId: null, // General warning, no checkbox
    check: (data) => data.road?.speed_limit >= 70,
    icon: '🚗',
    text: 'High Speed Road',
    reason: (data) => `Speed limit is ${data.road.speed_limit} km/h`
  },
  {
    id: 'night_work',
    hazardId: null,
    check: (data) => data.work?.shift === 'night',
    icon: '🌙',
    text: 'Night Work - Reduced Visibility',
    reason: () => 'Night shift selected'
  },
  {
    id: 'high_traffic',
    hazardId: 'tmp-hazard-high-traffic',
    check: (data) => {
      // High traffic based on AADT from API or high speed + multiple lanes
      const aadt = data.api_data?.ccc_traffic?.counts?.[0]?.adt || 
                   data.api_data?.onrc?.results?.[0]?.traffic_volume;
      if (aadt && aadt > 10000) return true;
      
      const isHighSpeed = data.road?.speed_limit >= 60;
      const isMultiLane = data.road?.lanes >= 2;
      return isHighSpeed && isMultiLane;
    },
    icon: '⚠️',
    text: 'High Traffic Impact',
    reason: (data) => {
      const aadt = data.api_data?.ccc_traffic?.counts?.[0]?.adt || 
                   data.api_data?.onrc?.results?.[0]?.traffic_volume;
      if (aadt && aadt > 10000) return `AADT is ${aadt.toLocaleString()} vehicles/day`;
      return 'High speed multi-lane road';
    }
  },
  {
    id: 'school_zone',
    hazardId: 'tmp-hazard-school',
    check: (data) => {
      // Check API data for nearby schools
      return data.api_data?.osm?.schools?.length > 0;
    },
    icon: '🏫',
    text: 'School Nearby',
    reason: (data) => `${data.api_data?.osm?.schools?.length} school(s) within search radius`
  },
  {
    id: 'hospital_nearby',
    hazardId: 'tmp-hazard-hospital',
    check: (data) => {
      return data.api_data?.osm?.hospitals?.length > 0;
    },
    icon: '🏥',
    text: 'Hospital/Medical Facility Nearby',
    reason: (data) => `${data.api_data?.osm?.hospitals?.length} medical facility(s) within search radius`
  },
  {
    id: 'pedestrian_impact',
    hazardId: null,
    check: (data) => {
      const hasFootpath = data.road?.has_footpath || data.api_data?.osm?.footpath;
      const affectsFootpath = data.aspects_affected?.footpath || data.aspects_affected?.pedestrians;
      return hasFootpath && affectsFootpath;
    },
    icon: '🚶',
    text: 'Pedestrian Route Affected',
    reason: () => 'Footpath present and affected by work'
  },
  {
    id: 'cyclist_impact',
    hazardId: null,
    check: (data) => {
      const hasCycle = data.road?.has_cycle_lane || data.api_data?.osm?.cycle_lane;
      const affectsCycle = data.aspects_affected?.cycle_lane;
      return hasCycle && affectsCycle;
    },
    icon: '🚴',
    text: 'Cycle Lane Affected',
    reason: () => 'Cycle lane present and affected by work'
  },
  {
    id: 'bus_impact',
    hazardId: null,
    check: (data) => {
      const hasBusStop = data.road?.has_bus_stop || data.api_data?.osm?.bus_stops?.length > 0;
      const affectsBus = data.aspects_affected?.bus_stop;
      return hasBusStop || affectsBus;
    },
    icon: '🚌',
    text: 'Bus Route/Stop Affected',
    reason: (data) => {
      const count = data.api_data?.osm?.bus_stops?.length;
      return count ? `${count} bus stop(s) nearby` : 'Bus stop nearby or affected';
    }
  },
  {
    id: 'traffic_signals',
    hazardId: null,
    check: (data) => {
      return data.aspects_affected?.traffic_signals || data.api_data?.osm?.traffic_signals?.length > 0;
    },
    icon: '🚦',
    text: 'Traffic Signals Nearby',
    reason: (data) => {
      const count = data.api_data?.osm?.traffic_signals?.length;
      return count ? `${count} signal(s) within 100m - may need RTO coordination` : 'Traffic signals affected';
    }
  },
  {
    id: 'rail_corridor',
    hazardId: null,
    check: (data) => data.api_data?.osm?.rail?.length > 0,
    icon: '🚂',
    text: 'Rail Corridor Nearby',
    reason: () => 'Rail line within 500m - check 5m setback requirements'
  },
  {
    id: 'restrictions',
    hazardId: null,
    check: (data) => data.api_data?.restrictions?.length > 0,
    icon: '⚠️',
    text: 'Route Restrictions',
    reason: (data) => data.api_data?.restrictions?.map(r => r.type).join(', ')
  }
];

// Collect form data for hazard analysis
function collectFormDataForHazards() {
  return {
    road: {
      speed_limit: parseInt(document.getElementById('tmp-speed-limit')?.value) || null,
      lanes: parseInt(document.getElementById('tmp-lanes')?.value) || null,
      has_footpath: document.getElementById('tmp-has-footpath')?.checked,
      has_cycle_lane: document.getElementById('tmp-has-cycle-lane')?.checked,
      has_bus_stop: document.getElementById('tmp-has-bus-stop')?.checked,
    },
    work: {
      shift: document.getElementById('tmp-shift')?.value,
    },
    aspects_affected: {
      footpath: document.getElementById('tmp-affects-footpath')?.checked,
      pedestrians: document.getElementById('tmp-affects-pedestrians')?.checked,
      cycle_lane: document.getElementById('tmp-affects-cycle')?.checked,
      bus_stop: document.getElementById('tmp-affects-bus')?.checked,
      traffic_signals: document.getElementById('tmp-affects-signals')?.checked,
    },
    // Include API data for additional context
    api_data: tmpApiData
  };
}

// Update hazard suggestions based on form data
function updateHazardSuggestions() {
  const formData = collectFormDataForHazards();
  const suggestions = [];
  
  for (const rule of HAZARD_RULES) {
    try {
      if (rule.check(formData)) {
        // Check if already applied (checkbox checked)
        const isApplied = rule.hazardId && document.getElementById(rule.hazardId)?.checked;
        
        suggestions.push({
          id: rule.id,
          hazardId: rule.hazardId,
          icon: rule.icon,
          text: rule.text,
          reason: rule.reason(formData),
          applied: isApplied
        });
      }
    } catch (e) {
      // Ignore errors from incomplete data
    }
  }
  
  // Show or hide suggestions panel
  if (suggestions.length > 0) {
    hazardSuggestionsDiv.style.display = 'block';
    renderSuggestions(suggestions);
  } else {
    hazardSuggestionsDiv.style.display = 'none';
  }
}

// Render suggestions in the UI
function renderSuggestions(suggestions) {
  suggestionsListDiv.innerHTML = suggestions.map(s => `
    <div class="suggestion-item ${s.applied ? 'applied' : ''}" data-id="${s.id}">
      <span class="suggestion-icon">${s.icon}</span>
      <div class="suggestion-text">
        <div>${s.text}</div>
        <div class="suggestion-reason">${s.reason}</div>
      </div>
      ${s.hazardId ? `
        <button class="btn-add" data-hazard="${s.hazardId}" ${s.applied ? 'disabled' : ''}>
          ${s.applied ? '✓' : 'Add'}
        </button>
      ` : ''}
    </div>
  `).join('');
  
  // Add click handlers for "Add" buttons
  suggestionsListDiv.querySelectorAll('.btn-add:not([disabled])').forEach(btn => {
    btn.addEventListener('click', (e) => {
      const hazardId = btn.dataset.hazard;
      const checkbox = document.getElementById(hazardId);
      if (checkbox) {
        checkbox.checked = true;
        updateHazardSuggestions(); // Refresh to show as applied
      }
    });
  });
}

// Setup listeners for form changes to update suggestions
function setupHazardSuggestionListeners() {
  // Listen to relevant form fields
  const fieldsToWatch = [
    'tmp-speed-limit', 'tmp-lanes', 'tmp-shift',
    'tmp-has-footpath', 'tmp-has-cycle-lane', 'tmp-has-bus-stop',
    'tmp-affects-footpath', 'tmp-affects-pedestrians', 'tmp-affects-cycle', 
    'tmp-affects-bus', 'tmp-affects-signals'
  ];
  
  fieldsToWatch.forEach(id => {
    const el = document.getElementById(id);
    if (el) {
      el.addEventListener('change', updateHazardSuggestions);
    }
  });
  
  // Dismiss suggestions button
  document.getElementById('dismiss-suggestions')?.addEventListener('click', () => {
    hazardSuggestionsDiv.style.display = 'none';
  });
}

// Initialize on DOM ready
document.addEventListener('DOMContentLoaded', () => {
  initPinnedLocation();
  setupHazardSuggestionListeners();
  
  // Initial check for suggestions (after a short delay to let form populate)
  setTimeout(updateHazardSuggestions, 500);
});

// ==================== NZTA API INTEGRATION ====================

const NZTA_SPEED_LIMIT_API = 'https://services.arcgis.com/CXBb7LAjgIIdcsPt/arcgis/rest/services/SpeedLimitZoneFull__View/FeatureServer/0/query';
const NZTA_TRAFFIC_MONITORING_API = 'https://services.arcgis.com/CXBb7LAjgIIdcsPt/arcgis/rest/services/Assets_SHTrafficMonitoringSites/FeatureServer/0/query';
const NZTA_ONE_NETWORK_API = 'https://services.arcgis.com/CXBb7LAjgIIdcsPt/arcgis/rest/services/ONRC__Network/FeatureServer/0/query';

// Fetch speed limit at coordinates
async function fetchSpeedLimit(lat, lon) {
  try {
    const url = new URL(NZTA_SPEED_LIMIT_API);
    url.searchParams.set('geometry', `${lon},${lat}`);
    url.searchParams.set('geometryType', 'esriGeometryPoint');
    url.searchParams.set('inSR', '4326');
    url.searchParams.set('spatialRel', 'esriSpatialRelIntersects');
    url.searchParams.set('outFields', 'speedLimitZoneValue,speedCategoryName,rcaZoneReferenceName');
    url.searchParams.set('f', 'json');
    
    const resp = await fetch(url);
    const data = await resp.json();
    
    if (data.features && data.features.length > 0) {
      const attrs = data.features[0].attributes;
      return {
        speedLimit: parseInt(attrs.speedLimitZoneValue) || null,
        category: attrs.speedCategoryName,
        rca: attrs.rcaZoneReferenceName
      };
    }
    return null;
  } catch (err) {
    console.error('Failed to fetch speed limit:', err);
    return null;
  }
}

// Fetch nearby traffic monitoring sites
async function fetchNearbyTrafficSites(lat, lon, radiusMeters = 5000) {
  try {
    const url = new URL(NZTA_TRAFFIC_MONITORING_API);
    url.searchParams.set('geometry', `${lon},${lat}`);
    url.searchParams.set('geometryType', 'esriGeometryPoint');
    url.searchParams.set('inSR', '4326');
    url.searchParams.set('distance', radiusMeters);
    url.searchParams.set('units', 'esriSRUnit_Meter');
    url.searchParams.set('spatialRel', 'esriSpatialRelIntersects');
    url.searchParams.set('outFields', 'siteref,description,aadt1yearago,region');
    url.searchParams.set('f', 'json');
    
    const resp = await fetch(url);
    const data = await resp.json();
    
    if (data.features && data.features.length > 0) {
      return data.features.map(f => ({
        siteRef: f.attributes.siteref,
        description: f.attributes.description,
        aadt: f.attributes.aadt1yearago,
        region: f.attributes.region
      }));
    }
    return [];
  } catch (err) {
    console.error('Failed to fetch traffic sites:', err);
    return [];
  }
}

// Fetch road classification (ONRC)
async function fetchRoadClassification(lat, lon) {
  try {
    const url = new URL(NZTA_ONE_NETWORK_API);
    url.searchParams.set('geometry', `${lon},${lat}`);
    url.searchParams.set('geometryType', 'esriGeometryPoint');
    url.searchParams.set('inSR', '4326');
    url.searchParams.set('distance', '50'); // 50m buffer
    url.searchParams.set('units', 'esriSRUnit_Meter');
    url.searchParams.set('spatialRel', 'esriSpatialRelIntersects');
    url.searchParams.set('outFields', 'onrc_category_name,road_name,rca_name');
    url.searchParams.set('f', 'json');
    
    const resp = await fetch(url);
    const data = await resp.json();
    
    if (data.features && data.features.length > 0) {
      const attrs = data.features[0].attributes;
      return {
        category: attrs.onrc_category_name,
        roadName: attrs.road_name,
        rca: attrs.rca_name
      };
    }
    return null;
  } catch (err) {
    console.error('Failed to fetch road classification:', err);
    return null;
  }
}

// ==================== AUTO-FILL FROM API ====================

// Container for API-fetched data
let apiData = {
  speedLimit: null,
  trafficSites: [],
  roadClassification: null,
  lastFetchCoords: null
};

// Show loading indicator
function showApiLoadingIndicator() {
  const indicator = document.getElementById('api-loading-indicator');
  if (indicator) indicator.style.display = 'flex';
}

// Hide loading indicator
function hideApiLoadingIndicator() {
  const indicator = document.getElementById('api-loading-indicator');
  if (indicator) indicator.style.display = 'none';
}

// Fetch all API data for a location
async function fetchLocationData(lat, lon) {
  // Don't refetch if same location
  const coordKey = `${lat.toFixed(5)},${lon.toFixed(5)}`;
  if (apiData.lastFetchCoords === coordKey) {
    return apiData;
  }
  
  showApiLoadingIndicator();
  
  try {
    // Fetch in parallel
    const [speedData, trafficSites, roadClass] = await Promise.all([
      fetchSpeedLimit(lat, lon),
      fetchNearbyTrafficSites(lat, lon, 5000),
      fetchRoadClassification(lat, lon)
    ]);
    
    apiData = {
      speedLimit: speedData,
      trafficSites: trafficSites,
      roadClassification: roadClass,
      lastFetchCoords: coordKey
    };
    
    // Update UI with fetched data
    updateFormWithApiData();
    showApiDataPanel();
    
  } catch (err) {
    console.error('Error fetching location data:', err);
  } finally {
    hideApiLoadingIndicator();
  }
  
  return apiData;
}

// Update form fields with API data
function updateFormWithApiData() {
  // Speed limit
  if (apiData.speedLimit?.speedLimit) {
    const speedSelect = document.getElementById('tmp-speed-limit');
    const speedValue = apiData.speedLimit.speedLimit.toString();
    
    // Check if value exists in dropdown
    const option = Array.from(speedSelect.options).find(o => o.value === speedValue);
    if (option) {
      speedSelect.value = speedValue;
      highlightAutoFilled(speedSelect);
    }
  }
  
  // Road name (if available and field is empty)
  if (apiData.roadClassification?.roadName) {
    const roadNameInput = document.getElementById('tmp-road-name');
    if (roadNameInput && !roadNameInput.value) {
      roadNameInput.value = apiData.roadClassification.roadName;
      highlightAutoFilled(roadNameInput);
    }
  }
  
  // Trigger hazard suggestions update
  updateHazardSuggestions();
}

// Visual highlight for auto-filled fields
function highlightAutoFilled(element) {
  element.classList.add('auto-filled');
  element.title = 'Auto-filled from NZTA data';
  
  // Remove highlight after user interaction
  element.addEventListener('change', () => {
    element.classList.remove('auto-filled');
    element.title = '';
  }, { once: true });
}

// Show panel with API data
function showApiDataPanel() {
  let panel = document.getElementById('api-data-panel');
  
  if (!panel) {
    // Create panel if it doesn't exist
    panel = document.createElement('div');
    panel.id = 'api-data-panel';
    panel.className = 'api-data-panel';
    
    // Insert after pinned location section
    const locationSection = document.querySelector('#tab-tmp .tmp-section:nth-child(3)');
    if (locationSection) {
      locationSection.appendChild(panel);
    }
  }
  
  // Build panel content
  let html = '<div class="api-data-header">📡 Data from NZTA</div>';
  
  // Speed limit
  if (apiData.speedLimit) {
    html += `
      <div class="api-data-item">
        <span class="api-data-label">Speed Limit:</span>
        <span class="api-data-value">${apiData.speedLimit.speedLimit} km/h</span>
        <span class="api-data-source">(${apiData.speedLimit.category || 'Permanent'})</span>
      </div>
    `;
  }
  
  // Road classification
  if (apiData.roadClassification) {
    html += `
      <div class="api-data-item">
        <span class="api-data-label">Road Class:</span>
        <span class="api-data-value">${apiData.roadClassification.category || 'Unknown'}</span>
      </div>
    `;
    if (apiData.roadClassification.rca) {
      html += `
        <div class="api-data-item">
          <span class="api-data-label">RCA:</span>
          <span class="api-data-value">${apiData.roadClassification.rca}</span>
        </div>
      `;
    }
  }
  
  // Traffic sites
  if (apiData.trafficSites && apiData.trafficSites.length > 0) {
    const nearestSite = apiData.trafficSites[0];
    html += `
      <div class="api-data-item">
        <span class="api-data-label">Nearest Traffic Site:</span>
        <span class="api-data-value">${nearestSite.description || nearestSite.siteRef}</span>
      </div>
    `;
    if (nearestSite.aadt) {
      html += `
        <div class="api-data-item">
          <span class="api-data-label">AADT:</span>
          <span class="api-data-value">${nearestSite.aadt.toLocaleString()} vehicles/day</span>
        </div>
      `;
    }
  }
  
  if (html === '<div class="api-data-header">📡 Data from NZTA</div>') {
    html += '<div class="api-data-item"><span class="api-data-value">No data found for this location</span></div>';
  }
  
  panel.innerHTML = html;
  panel.style.display = 'block';
}

// ==================== TMP API INTEGRATION ====================

const TMP_API_URL = 'https://tmp-api.thelockmeister.workers.dev/api/tmp-data';

// Store the latest API response
let tmpApiData = null;

// Get selected search radius
function getSearchRadius() {
  const radiusSelect = document.getElementById('search-radius');
  return radiusSelect ? parseInt(radiusSelect.value) : 500;
}

// Refresh data with current radius
document.getElementById('refresh-data-btn')?.addEventListener('click', async () => {
  const lat = pinnedLocation?.lat || currentParams?.lat;
  const lng = pinnedLocation?.lng || currentParams?.lng;
  
  if (!lat || !lng) {
    alert('Navigate to a Street View location first.');
    return;
  }
  
  // Clear cached data to force refetch
  tmpApiData = null;
  await fetchFromTmpApi(lat, lng);
});

// Fetch all data from our unified TMP API
async function fetchFromTmpApi(lat, lng) {
  const radius = getSearchRadius();
  console.log('Fetching from TMP API for:', lat, lng, 'radius:', radius);
  showApiLoadingIndicator();
  
  try {
    const url = `${TMP_API_URL}?lat=${lat}&lng=${lng}&radius=${radius}`;
    const response = await fetch(url);
    
    if (!response.ok) {
      throw new Error(`TMP API returned ${response.status}`);
    }
    
    tmpApiData = await response.json();
    console.log('TMP API response:', tmpApiData);
    
    // Display the fetched data
    displayFetchedData(tmpApiData);
    
    // Enable autofill buttons
    const autofillAspectsBtn = document.getElementById('autofill-aspects-btn');
    if (autofillAspectsBtn) autofillAspectsBtn.disabled = false;
    
    return tmpApiData;
  } catch (err) {
    console.error('TMP API error:', err);
    return null;
  } finally {
    hideApiLoadingIndicator();
  }
}

// Navigate to location in Google Maps with a marker
// If newTab=true or no Maps tab exists, opens new tab
// Otherwise navigates in existing Maps tab (will reload)
async function openGoogleMapsAt(lat, lng, label, newTab = false) {
  // Use place/search format to show a red marker pin
  // Encode the label for the search query, fallback to coords
  const searchQuery = encodeURIComponent(label || `${lat},${lng}`);
  const mapUrl = `https://www.google.com/maps/place/${searchQuery}/@${lat},${lng},18z/data=!4m2!3m1!1s0x0:0x0?entry=ttu`;
  
  // Alternative: direct marker URL format
  const markerUrl = `https://www.google.com/maps/search/${lat},${lng}/@${lat},${lng},18z`;
  
  if (newTab) {
    chrome.tabs.create({ url: markerUrl });
    return;
  }
  
  try {
    // Find the active Google Maps tab
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    
    if (tab && (tab.url?.includes('google.com/maps') || tab.url?.includes('maps.google.com'))) {
      // Navigate in same tab
      await chrome.scripting.executeScript({
        target: { tabId: tab.id },
        func: (url) => { window.location.href = url; },
        args: [markerUrl]
      });
    } else {
      // No Maps tab active, open new
      chrome.tabs.create({ url: markerUrl });
    }
  } catch (err) {
    console.error('Error navigating to location:', err);
    chrome.tabs.create({ url: markerUrl });
  }
}

// Render a collapsible list of clickable location items
function renderLocationItems(items, labelKey, latKey, lngKey, listId) {
  if (!items || items.length === 0) return '';
  
  const itemsHtml = items.map((item, i) => {
    const lat = item[latKey] || item.lat;
    const lng = item[lngKey] || item.lng || item.lon;
    const label = item[labelKey] || item.name || `Item ${i + 1}`;
    const distance = item.distance_m ? `${Math.round(item.distance_m)}m` : (item.distance ? `${Math.round(item.distance)}m` : '');
    
    if (!lat || !lng) return '';
    
    return `
      <div class="fetched-location-item" data-lat="${lat}" data-lng="${lng}" data-label="${escapeHtml(label)}">
        <span class="location-name">${escapeHtml(label)}</span>
        ${distance ? `<span class="location-distance">${distance}</span>` : ''}
        <span class="location-action">📍</span>
      </div>
    `;
  }).join('');
  
  return `
    <div class="fetched-locations collapsible" id="${listId}">
      ${itemsHtml}
    </div>
  `;
}

// Render collapsible section header
function renderCollapsibleHeader(icon, label, count, listId) {
  return `
    <div class="fetched-item collapsible-header" data-target="${listId}">
      <span class="fetched-label">${icon} ${label} (${count}):</span>
      <span class="collapse-toggle">▼</span>
    </div>
  `;
}

// Display fetched data in the UI
function displayFetchedData(data) {
  const section = document.getElementById('fetched-data-section');
  const content = document.getElementById('fetched-data-content');
  
  if (!section || !content || !data) return;
  
  const radius = getSearchRadius();
  let html = `<div class="fetched-radius-note">Data within ${radius}m radius • Click to view, Shift+click for new tab</div>`;
  
  // Speed limit
  if (data.nzta_speed?.speed_limit) {
    html += `
      <div class="fetched-item">
        <span class="fetched-label">Speed Limit:</span>
        <span class="fetched-value">${data.nzta_speed.speed_limit} km/h</span>
        <span class="fetched-source">(NZTA - ${data.nzta_speed.speed_category || 'Permanent'})</span>
      </div>
    `;
  }
  
  // Road hierarchy (CCC)
  if (data.ccc_hierarchy?.results?.length > 0) {
    const rec = data.ccc_hierarchy.results[data.ccc_hierarchy.recommended || 0];
    if (rec) {
      html += `
        <div class="fetched-item">
          <span class="fetched-label">Road Hierarchy:</span>
          <span class="fetched-value">${rec.hierarchy || 'Unknown'}</span>
          <span class="fetched-source">(CCC)</span>
        </div>
      `;
    }
  }
  
  // ONRC classification
  if (data.onrc?.results?.length > 0) {
    const rec = data.onrc.results[data.onrc.recommended || 0];
    if (rec) {
      html += `
        <div class="fetched-item">
          <span class="fetched-label">ONRC:</span>
          <span class="fetched-value">${rec.classification || 'Unknown'}</span>
          <span class="fetched-source">(NZTA)</span>
        </div>
      `;
      if (rec.traffic_volume) {
        html += `
          <div class="fetched-item">
            <span class="fetched-label">Traffic Volume:</span>
            <span class="fetched-value">${rec.traffic_volume.toLocaleString()} vpd</span>
          </div>
        `;
      }
    }
  }
  
  // CCC Traffic counts
  if (data.ccc_traffic?.counts?.length > 0) {
    const rec = data.ccc_traffic.counts[data.ccc_traffic.recommended || 0];
    if (rec) {
      html += `
        <div class="fetched-item">
          <span class="fetched-label">AADT:</span>
          <span class="fetched-value">${rec.adt?.toLocaleString() || 'N/A'} vpd</span>
          <span class="fetched-source">(CCC - ${rec.count_date || 'Unknown date'})</span>
        </div>
      `;
      if (rec.pct_heavy) {
        html += `
          <div class="fetched-item">
            <span class="fetched-label">Heavy Vehicles:</span>
            <span class="fetched-value">${rec.pct_heavy}%</span>
          </div>
        `;
      }
    }
  }
  
  // Road name - prefer ONRC, then CCC hierarchy, then OSM
  const roadName = data.onrc?.results?.[0]?.road_name || 
                   data.ccc_hierarchy?.results?.[0]?.road_name || 
                   data.osm?.road_name;
  const roadNameSource = data.onrc?.results?.[0]?.road_name ? 'NZTA ONRC' :
                         data.ccc_hierarchy?.results?.[0]?.road_name ? 'CCC' : 'OSM';
  if (roadName) {
    html += `
      <div class="fetched-item">
        <span class="fetched-label">Road Name:</span>
        <span class="fetched-value">${roadName}</span>
        <span class="fetched-source">(${roadNameSource})</span>
      </div>
    `;
  }
  
  // Features summary
  if (data.osm) {
    const features = [];
    if (data.osm.footpath || data.osm.has_footpath) features.push('Footpath');
    if (data.osm.cycle_lane || data.osm.has_cycle_lane) features.push('Cycle Lane');
    
    if (features.length > 0) {
      html += `
        <div class="fetched-item">
          <span class="fetched-label">Road Features:</span>
          <span class="fetched-value">${features.join(', ')}</span>
        </div>
      `;
    }
  }
  
  // Schools - collapsible clickable list
  if (data.osm?.schools?.length > 0) {
    html += renderCollapsibleHeader('🏫', 'Schools', data.osm.schools.length, 'schools-list');
    html += renderLocationItems(data.osm.schools, 'name', 'lat', 'lon', 'schools-list');
  }
  
  // Hospitals - collapsible clickable list
  if (data.osm?.hospitals?.length > 0) {
    html += renderCollapsibleHeader('🏥', 'Hospitals/Medical', data.osm.hospitals.length, 'hospitals-list');
    html += renderLocationItems(data.osm.hospitals, 'name', 'lat', 'lon', 'hospitals-list');
  }
  
  // Bus stops - collapsible clickable list
  if (data.osm?.bus_stops?.length > 0) {
    html += renderCollapsibleHeader('🚌', 'Bus Stops', data.osm.bus_stops.length, 'bus-stops-list');
    html += renderLocationItems(data.osm.bus_stops, 'name', 'lat', 'lon', 'bus-stops-list');
  }
  
  // Traffic signals - collapsible clickable list
  if (data.osm?.traffic_signals?.length > 0) {
    html += renderCollapsibleHeader('🚦', 'Traffic Signals', data.osm.traffic_signals.length, 'signals-list');
    html += renderLocationItems(data.osm.traffic_signals, 'name', 'lat', 'lon', 'signals-list');
  }
  
  // Rail corridors - collapsible clickable
  if (data.osm?.rail?.length > 0) {
    html += renderCollapsibleHeader('🚂', 'Rail Corridors', data.osm.rail.length, 'rail-list');
    html += renderLocationItems(data.osm.rail, 'name', 'lat', 'lon', 'rail-list');
  }
  
  // Restrictions
  if (data.restrictions?.length > 0) {
    html += `
      <div class="fetched-item fetched-warning">
        <span class="fetched-label">⚠️ Restrictions:</span>
        <span class="fetched-value">${data.restrictions.map(r => r.type).join(', ')}</span>
      </div>
    `;
  }
  
  if (html) {
    content.innerHTML = html;
    section.style.display = 'block';
    
    // Add click handlers for location items
    // Shift+click opens in new tab, regular click navigates same tab
    content.querySelectorAll('.fetched-location-item').forEach(item => {
      item.addEventListener('click', (e) => {
        const lat = item.dataset.lat;
        const lng = item.dataset.lng;
        const label = item.dataset.label;
        openGoogleMapsAt(lat, lng, label, e.shiftKey);
      });
    });
    
    // Add collapsible toggle handlers
    content.querySelectorAll('.collapsible-header').forEach(header => {
      header.addEventListener('click', () => {
        const targetId = header.dataset.target;
        const target = document.getElementById(targetId);
        const toggle = header.querySelector('.collapse-toggle');
        
        if (target) {
          const isCollapsed = target.classList.toggle('collapsed');
          if (toggle) toggle.textContent = isCollapsed ? '▶' : '▼';
        }
      });
    });
  } else {
    section.style.display = 'none';
  }
}

// Auto-fill aspects affected based on API data
function autofillAspectsAffected() {
  if (!tmpApiData) return;
  
  // Check footpath
  if (tmpApiData.osm?.footpath) {
    document.getElementById('tmp-has-footpath').checked = true;
  }
  
  // Check cycle lane
  if (tmpApiData.osm?.cycle_lane) {
    document.getElementById('tmp-has-cycle-lane').checked = true;
  }
  
  // Check bus stops
  if (tmpApiData.osm?.bus_stops?.length > 0) {
    document.getElementById('tmp-has-bus-stop').checked = true;
  }
  
  // Check traffic signals nearby
  if (tmpApiData.osm?.traffic_signals?.length > 0) {
    document.getElementById('tmp-affects-signals').checked = true;
  }
  
  // Check schools nearby - suggest hazard
  if (tmpApiData.osm?.schools?.length > 0) {
    document.getElementById('tmp-hazard-school').checked = true;
  }
  
  // If footpath present, likely affects pedestrians
  if (tmpApiData.osm?.footpath) {
    document.getElementById('tmp-affects-pedestrians').checked = true;
    document.getElementById('tmp-affects-footpath').checked = true;
  }
  
  // If cycle lane present, affects cyclists
  if (tmpApiData.osm?.cycle_lane) {
    document.getElementById('tmp-affects-cycle').checked = true;
  }
  
  // If bus stops, affects bus
  if (tmpApiData.osm?.bus_stops?.length > 0) {
    document.getElementById('tmp-affects-bus').checked = true;
  }
  
  // High traffic based on AADT
  const aadt = tmpApiData.ccc_traffic?.counts?.[0]?.adt || tmpApiData.onrc?.results?.[0]?.traffic_volume;
  if (aadt && aadt > 10000) {
    document.getElementById('tmp-hazard-high-traffic').checked = true;
  }
  
  // Update hazard suggestions
  updateHazardSuggestions();
}

// Setup autofill aspects button
document.getElementById('autofill-aspects-btn')?.addEventListener('click', autofillAspectsAffected);

// Dismiss fetched data section
document.getElementById('dismiss-fetched-data')?.addEventListener('click', () => {
  document.getElementById('fetched-data-section').style.display = 'none';
});

// Hook into pin location to trigger API fetch and address lookup
const originalPinCurrentLocation = pinCurrentLocation;
pinCurrentLocation = async function() {
  originalPinCurrentLocation();
  
  if (pinnedLocation) {
    // Fetch TMP API data and address in parallel
    await Promise.all([
      fetchFromTmpApi(pinnedLocation.lat, pinnedLocation.lng),
      fetchLocationAddress(pinnedLocation.lat, pinnedLocation.lng, true) // force update
    ]);
  }
};

// Also fetch when loading a saved session with coordinates
const originalLoadTmpSession = loadTmpSession;
loadTmpSession = async function(sessionId) {
  await originalLoadTmpSession(sessionId);
  
  if (currentTmpData?.coordinates) {
    await fetchLocationData(currentTmpData.coordinates.lat, currentTmpData.coordinates.lng);
  }
};

// ==================== AUTOFILL ROAD DATA BUTTON ==================== 

// Enable autofill button when in Street View
const autofillRoadBtn = document.getElementById('autofill-road-btn');
const roadDataComparison = document.getElementById('road-data-comparison');
const comparisonContent = document.getElementById('comparison-content');

// Track fetched road data from multiple sources
let fetchedRoadData = {
  speedLimit: [],
  lanes: [],
  roadLevel: [],
  roadName: []
};

// Enable buttons when Street View active
const autofillLocationBtn = document.getElementById('autofill-location-btn');

// Update autofill button states based on available location
function updateAutofillButtonStates() {
  const hasLocation = !!(pinnedLocation?.lat || currentParams?.lat);
  if (autofillRoadBtn) autofillRoadBtn.disabled = !hasLocation;
  if (autofillLocationBtn) autofillLocationBtn.disabled = !hasLocation;
  
  const refreshBtn = document.getElementById('refresh-data-btn');
  if (refreshBtn) refreshBtn.disabled = !hasLocation;
}

const originalSetStreetViewState = setStreetViewState;
setStreetViewState = function() {
  originalSetStreetViewState();
  updateAutofillButtonStates();
};

const originalSetDisconnectedState = setDisconnectedState;
setDisconnectedState = function() {
  originalSetDisconnectedState();
  updateAutofillButtonStates();
};

const originalSetConnectedState = setConnectedState;
setConnectedState = function() {
  originalSetConnectedState();
  updateAutofillButtonStates();
};

// Also update when pin changes
const originalUpdatePinnedLocationUI = updatePinnedLocationUI;
updatePinnedLocationUI = function() {
  originalUpdatePinnedLocationUI();
  updateAutofillButtonStates();
};

// Autofill location button click handler - uses TMP API + reverse geocoding
autofillLocationBtn?.addEventListener('click', async () => {
  // Use pinned location if available, otherwise current view
  const lat = pinnedLocation?.lat || currentParams?.lat;
  const lng = pinnedLocation?.lng || currentParams?.lng;
  
  if (!lat || !lng) {
    alert('Navigate to a Street View location first.');
    return;
  }
  
  autofillLocationBtn.disabled = true;
  autofillLocationBtn.textContent = '⏳ Fetching...';
  
  try {
    // Fetch from TMP API (will also populate fetched data panel)
    const apiData = await fetchFromTmpApi(lat, lng);
    
    // Also do reverse geocoding for address details
    await fetchLocationAddress(lat, lng);
    
    autofillLocationBtn.textContent = '✓ Filled!';
    setTimeout(() => {
      autofillLocationBtn.textContent = '🔍 Auto-Fill Location';
      autofillLocationBtn.disabled = false;
    }, 2000);
  } catch (err) {
    console.error('Error in autofill location:', err);
    autofillLocationBtn.textContent = '🔍 Auto-Fill Location';
    autofillLocationBtn.disabled = false;
  }
});

// Autofill road data button click handler - now uses TMP API
autofillRoadBtn?.addEventListener('click', async () => {
  // Use pinned location if available, otherwise current view
  const lat = pinnedLocation?.lat || currentParams?.lat;
  const lng = pinnedLocation?.lng || currentParams?.lng;
  
  if (!lat || !lng) {
    alert('Navigate to a Street View location first.');
    return;
  }
  
  autofillRoadBtn.disabled = true;
  autofillRoadBtn.textContent = '⏳ Fetching...';
  
  try {
    // Use the unified TMP API
    const data = await fetchFromTmpApi(lat, lng);
    
    if (data) {
      // Auto-apply road data from API response
      applyRoadDataFromApi(data);
    }
  } finally {
    autofillRoadBtn.disabled = false;
    autofillRoadBtn.textContent = '🔍 Auto-Fill Road Data';
  }
});

// Apply road characteristics from TMP API data
function applyRoadDataFromApi(data) {
  if (!data) return;
  
  // Speed limit from NZTA
  if (data.nzta_speed?.speed_limit) {
    const speedSelect = document.getElementById('tmp-speed-limit');
    const speedValue = data.nzta_speed.speed_limit.toString();
    if (Array.from(speedSelect.options).some(o => o.value === speedValue)) {
      speedSelect.value = speedValue;
      highlightAutoFilled(speedSelect);
    }
  }
  
  // Road name from OSM or ONRC
  const roadName = data.osm?.road_name || data.onrc?.results?.[0]?.road_name;
  if (roadName) {
    const roadNameInput = document.getElementById('tmp-road-name');
    if (roadNameInput && !roadNameInput.value) {
      roadNameInput.value = roadName;
      highlightAutoFilled(roadNameInput);
    }
  }
  
  // Road level from CCC hierarchy
  if (data.ccc_hierarchy?.results?.length > 0) {
    const hierarchy = data.ccc_hierarchy.results[data.ccc_hierarchy.recommended || 0]?.hierarchy;
    if (hierarchy) {
      const levelMap = {
        'Major Arterial': '1',
        'Minor Arterial': '1',
        'Collector': '2',
        'Local': '3',
        'Access': '4'
      };
      const level = Object.entries(levelMap).find(([k]) => hierarchy.includes(k))?.[1];
      if (level) {
        const roadLevelSelect = document.getElementById('tmp-road-level');
        roadLevelSelect.value = level;
        highlightAutoFilled(roadLevelSelect);
      }
    }
  }
  
  // Features present checkboxes
  if (data.osm?.footpath) {
    document.getElementById('tmp-has-footpath').checked = true;
  }
  if (data.osm?.cycle_lane) {
    document.getElementById('tmp-has-cycle-lane').checked = true;
  }
  if (data.osm?.bus_stops?.length > 0) {
    document.getElementById('tmp-has-bus-stop').checked = true;
  }
  if (data.osm?.traffic_signals?.length > 0) {
    document.getElementById('tmp-has-intersection').checked = true;
  }
  
  // Update hazard suggestions with new data
  updateHazardSuggestions();
}

// Fetch location address via reverse geocoding
// forceUpdate: if true, overwrite existing values
async function fetchLocationAddress(lat, lng, forceUpdate = false) {
  if (autofillLocationBtn) {
    autofillLocationBtn.disabled = true;
    autofillLocationBtn.textContent = '⏳ Fetching...';
  }
  
  try {
    // Reverse geocode using Nominatim (OpenStreetMap)
    const osmUrl = `https://nominatim.openstreetmap.org/reverse?format=json&lat=${lat}&lon=${lng}&zoom=18&addressdetails=1`;
    const osmResponse = await fetch(osmUrl);
    const osmData = await osmResponse.json();
    
    console.log('Location data from OSM:', osmData);
    
    // Extract address components
    const address = osmData.address || {};
    
    // Helper to set field value
    const setField = (id, value) => {
      const el = document.getElementById(id);
      if (el && value && (forceUpdate || !el.value)) {
        el.value = value;
        highlightAutoFilled(el);
      }
    };
    
    // Road name
    setField('tmp-road-name', address.road);
    
    // Full address - clean up the display name
    if (osmData.display_name) {
      const cleanAddress = osmData.display_name
        .replace(/, New Zealand$/, '')
        .replace(/, \d{4}/, '');
      setField('tmp-address', cleanAddress);
    }
    
    // Suburb
    const suburb = address.suburb || address.neighbourhood || address.village;
    setField('tmp-suburb', suburb);
    
    // City
    const city = address.city || address.town || address.municipality;
    setField('tmp-city', city);
    
    if (autofillLocationBtn) {
      autofillLocationBtn.textContent = '✓ Filled!';
      setTimeout(() => {
        autofillLocationBtn.textContent = '🔍 Auto-Fill Location';
        autofillLocationBtn.disabled = false;
      }, 2000);
    }
    
  } catch (err) {
    console.error('Error fetching location:', err);
    if (autofillLocationBtn) {
      autofillLocationBtn.textContent = '🔍 Auto-Fill Location';
      autofillLocationBtn.disabled = false;
    }
  }
}

// Fetch road data from all sources
async function fetchAllRoadData(lat, lng) {
  autofillRoadBtn.disabled = true;
  autofillRoadBtn.textContent = '⏳ Fetching...';
  
  // Reset fetched data
  fetchedRoadData = {
    speedLimit: [],
    lanes: [],
    roadLevel: [],
    roadName: []
  };
  
  try {
    // Prioritize NZTA (official source), only use OSM if NZTA fails
    await fetchFromNZTA(lat, lng);
    
    // Only fetch from OSM if we're missing critical data
    const needsOSM = !fetchedRoadData.lanes.length || !fetchedRoadData.roadName.length;
    if (needsOSM) {
      console.log('Fetching from OSM as fallback for missing data');
      await fetchFromOSM(lat, lng);
    }
    
    // Log results for debugging
    console.log('Fetched road data:', fetchedRoadData);
    
    // Show comparison panel
    renderDataComparison();
    roadDataComparison.style.display = 'block';
    
  } catch (err) {
    console.error('Error fetching road data:', err);
    alert('Failed to fetch road data: ' + err.message);
  } finally {
    autofillRoadBtn.disabled = false;
    autofillRoadBtn.textContent = '🔍 Auto-Fill Road Data';
  }
}

// Fetch from NZTA APIs
async function fetchFromNZTA(lat, lng) {
  console.log('Fetching from NZTA for:', lat, lng);
  try {
    // Speed limit
    const speedData = await fetchSpeedLimit(lat, lng);
    console.log('NZTA speed data:', speedData);
    if (speedData?.speedLimit) {
      fetchedRoadData.speedLimit.push({
        source: 'NZTA',
        value: speedData.speedLimit.toString(),
        confidence: 'high',
        metadata: speedData.category
      });
    }
    
    // Road classification (for road level estimation)
    const roadClass = await fetchRoadClassification(lat, lng);
    console.log('NZTA road classification:', roadClass);
    if (roadClass?.category) {
      if (roadClass.roadName) {
        fetchedRoadData.roadName.push({
          source: 'NZTA ONRC',
          value: roadClass.roadName,
          confidence: 'high'
        });
      }
      
      // Estimate road level from ONRC category
      const roadLevel = estimateRoadLevelFromONRC(roadClass.category);
      console.log('NZTA road level estimated:', roadLevel, 'from category:', roadClass.category);
      if (roadLevel) {
        fetchedRoadData.roadLevel.push({
          source: 'NZTA ONRC',
          value: roadLevel.toString(),
          confidence: 'medium',
          metadata: roadClass.category
        });
      }
      
      // Estimate lanes from ONRC category
      const lanes = estimateLanesFromONRC(roadClass.category);
      console.log('NZTA lanes estimated:', lanes, 'from category:', roadClass.category);
      if (lanes) {
        fetchedRoadData.lanes.push({
          source: 'NZTA ONRC',
          value: lanes.toString(),
          confidence: 'low',
          metadata: roadClass.category
        });
      }
    }
  } catch (err) {
    console.error('NZTA fetch error:', err);
  }
}

// Fetch from OpenStreetMap Nominatim (reverse geocoding) 
async function fetchFromOSM(lat, lng) {
  console.log('Fetching from OSM Nominatim for:', lat, lng);
  try {
    // Use Nominatim reverse geocoding (more reliable than Overpass for extensions)
    const url = `https://nominatim.openstreetmap.org/reverse?format=json&lat=${lat}&lon=${lng}&zoom=18&addressdetails=1&extratags=1`;
    
    const response = await fetch(url, {
      headers: {
        'User-Agent': 'TMP-Extension/2.0'
      }
    });
    
    console.log('OSM Nominatim response status:', response.status);
    
    if (!response.ok) {
      throw new Error('OSM Nominatim failed: ' + response.status);
    }
    
    const data = await response.json();
    console.log('OSM Nominatim data received:', data);
    
    const address = data.address || {};
    const extratags = data.extratags || {};
    
    console.log('OSM address:', address);
    console.log('OSM extratags:', extratags);
    
    // Road name
    if (address.road) {
      console.log('OSM road name found:', address.road);
      fetchedRoadData.roadName.push({
        source: 'OpenStreetMap',
        value: address.road,
        confidence: 'high'
      });
    }
    
    // Speed limit from extratags
    if (extratags.maxspeed) {
      const speed = parseInt(extratags.maxspeed);
      console.log('OSM speed limit found:', speed);
      if (!isNaN(speed)) {
        fetchedRoadData.speedLimit.push({
          source: 'OpenStreetMap',
          value: speed.toString(),
          confidence: 'medium'
        });
      }
    }
    
    // Lanes from extratags
    if (extratags.lanes) {
      const lanes = parseInt(extratags.lanes);
      console.log('OSM lanes found:', lanes);
      if (!isNaN(lanes)) {
        fetchedRoadData.lanes.push({
          source: 'OpenStreetMap',
          value: lanes.toString(),
          confidence: 'high'
        });
      }
    }
    
    // Estimate road level from highway type
    const highway = extratags.highway || data.type;
    if (highway) {
      const roadLevel = estimateRoadLevelFromOSMHighway(highway);
      console.log('OSM road level estimated:', roadLevel, 'from highway:', highway);
      if (roadLevel) {
        fetchedRoadData.roadLevel.push({
          source: 'OpenStreetMap',
          value: roadLevel.toString(),
          confidence: 'low',
          metadata: highway
        });
      }
    }
  } catch (err) {
    console.error('OSM fetch error:', err);
    // Don't throw - let other sources continue
  }
}

// Fetch from Google Maps Directions API (via reverse geocoding)
async function fetchFromGoogleMaps(lat, lng) {
  try {
    // Use Google's reverse geocoding to get address components
    // Note: This is limited without an API key, but can extract road name
    const url = `https://maps.googleapis.com/maps/api/geocode/json?latlng=${lat},${lng}&result_type=street_address`;
    
    const response = await fetch(url);
    if (!response.ok) return;
    
    const data = await response.json();
    
    if (data.results && data.results.length > 0) {
      const result = data.results[0];
      
      // Extract road name from address components
      const routeComponent = result.address_components?.find(c => c.types.includes('route'));
      if (routeComponent) {
        fetchedRoadData.roadName.push({
          source: 'Google Maps',
          value: routeComponent.long_name,
          confidence: 'high'
        });
      }
    }
  } catch (err) {
    // Google API might be blocked or require key, fail silently
    console.error('Google Maps fetch error:', err);
  }
}

// Estimate road level from ONRC category
function estimateRoadLevelFromONRC(category) {
  if (!category) return null;
  
  const cat = category.toLowerCase();
  
  // ONRC categories: National, Arterial, Primary Collector, Secondary Collector, Access, Low Volume
  if (cat.includes('national') || cat.includes('arterial')) return 1;
  if (cat.includes('primary collector')) return 2;
  if (cat.includes('secondary collector')) return 3;
  if (cat.includes('access') || cat.includes('low volume')) return 4;
  
  return null;
}

// Estimate lanes from ONRC category
function estimateLanesFromONRC(category) {
  if (!category) return null;
  
  const cat = category.toLowerCase();
  
  // Very rough estimates
  if (cat.includes('national')) return 4;
  if (cat.includes('arterial')) return 3;
  if (cat.includes('collector')) return 2;
  if (cat.includes('access') || cat.includes('low volume')) return 2;
  
  return null;
}

// Estimate road level from OSM highway type
function estimateRoadLevelFromOSMHighway(highway) {
  if (!highway) return null;
  
  const typeMap = {
    'motorway': 1,
    'trunk': 1,
    'primary': 2,
    'secondary': 2,
    'tertiary': 3,
    'residential': 4,
    'unclassified': 4,
    'service': 4
  };
  
  return typeMap[highway] || null;
}

// Render the comparison panel
function renderDataComparison() {
  const fields = [
    { key: 'speedLimit', label: 'Speed Limit', unit: 'km/h' },
    { key: 'lanes', label: 'Number of Lanes', unit: '' },
    { key: 'roadLevel', label: 'Road Level', unit: '', formatter: formatRoadLevel },
    { key: 'roadName', label: 'Road Name', unit: '' }
  ];
  
  let html = '';
  
  for (const field of fields) {
    const data = fetchedRoadData[field.key];
    
    if (!data || data.length === 0) continue;
    
    // Check for conflicts
    const uniqueValues = [...new Set(data.map(d => d.value))];
    const hasConflict = uniqueValues.length > 1;
    
    // Pick recommended value (highest confidence)
    const recommended = data.reduce((best, current) => {
      const confidenceOrder = { high: 3, medium: 2, low: 1 };
      return (confidenceOrder[current.confidence] || 0) > (confidenceOrder[best.confidence] || 0) 
        ? current 
        : best;
    });
    
    html += `
      <div class="comparison-field">
        <div class="comparison-field-name">
          ${field.label}
          ${hasConflict ? '<span style="color: #f57c00; margin-left: 4px;">⚠️ Conflict</span>' : ''}
        </div>
        <div class="comparison-sources">
    `;
    
    for (const item of data) {
      const isRecommended = item === recommended;
      const displayValue = field.formatter ? field.formatter(item.value) : item.value;
      const classes = [
        'comparison-source',
        isRecommended ? 'recommended' : '',
        hasConflict ? 'conflict' : 'match'
      ].filter(Boolean).join(' ');
      
      html += `
        <div class="${classes}">
          <div>
            <span class="source-label">${item.source}</span>
            <span class="source-value">${displayValue}${field.unit ? ' ' + field.unit : ''}</span>
            ${item.metadata ? `<span class="source-confidence">(${item.metadata})</span>` : ''}
          </div>
          <button class="apply-value-btn" data-field="${field.key}" data-value="${item.value}">
            ${isRecommended ? '✓ Apply' : 'Apply'}
          </button>
        </div>
      `;
    }
    
    html += `
        </div>
        ${hasConflict ? '<div class="comparison-note">Multiple sources disagree - review carefully</div>' : ''}
      </div>
    `;
  }
  
  if (html === '') {
    html = '<div class="comparison-note">No road data found from any source</div>';
  }
  
  comparisonContent.innerHTML = html;
  
  // Add click handlers to individual apply buttons
  comparisonContent.querySelectorAll('.apply-value-btn').forEach(btn => {
    btn.addEventListener('click', () => {
      const field = btn.dataset.field;
      const value = btn.dataset.value;
      applyFieldValue(field, value);
      btn.textContent = '✓';
      btn.classList.add('applied');
    });
  });
}

// Format road level for display
function formatRoadLevel(level) {
  const labels = {
    '1': 'Level 1 (High Volume)',
    '2': 'Level 2 (Medium)',
    '3': 'Level 3 (Low)',
    '4': 'Level 4 (Very Low)'
  };
  return labels[level] || level;
}

// Apply a single field value
function applyFieldValue(field, value) {
  const fieldMap = {
    speedLimit: 'tmp-speed-limit',
    lanes: 'tmp-lanes',
    roadLevel: 'tmp-road-level',
    roadName: 'tmp-road-name'
  };
  
  const elementId = fieldMap[field];
  const element = document.getElementById(elementId);
  
  if (element) {
    element.value = value;
    highlightAutoFilled(element);
    
    // Trigger change event for hazard suggestions
    element.dispatchEvent(new Event('change'));
  }
}

// Apply all recommended values
document.getElementById('apply-all-btn')?.addEventListener('click', () => {
  // Apply the recommended (highest confidence) value for each field
  for (const [key, data] of Object.entries(fetchedRoadData)) {
    if (data && data.length > 0) {
      const recommended = data.reduce((best, current) => {
        const confidenceOrder = { high: 3, medium: 2, low: 1 };
        return (confidenceOrder[current.confidence] || 0) > (confidenceOrder[best.confidence] || 0) 
          ? current 
          : best;
      });
      
      applyFieldValue(key, recommended.value);
    }
  }
  
  // Hide comparison panel
  roadDataComparison.style.display = 'none';
  
  // Update hazard suggestions
  updateHazardSuggestions();
});

// Dismiss comparison panel
document.getElementById('apply-none-btn')?.addEventListener('click', () => {
  roadDataComparison.style.display = 'none';
});

document.getElementById('dismiss-comparison')?.addEventListener('click', () => {
  roadDataComparison.style.display = 'none';
});

// Display version from manifest
chrome.runtime.getManifest && (() => {
  const manifest = chrome.runtime.getManifest();
  const versionEl = document.getElementById('version-display');
  if (versionEl && manifest.version) {
    versionEl.textContent = `v${manifest.version}`;
  }
})();
